(* Content-type: application/vnd.wolfram.mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 13.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       158,          7]
NotebookDataLength[    102436,       2603]
NotebookOptionsPosition[     89572,       2415]
NotebookOutlinePosition[     89969,       2431]
CellTagsIndexPosition[     89926,       2428]
WindowFrame->Normal*)

(* Beginning of Notebook Content *)
Notebook[{

Cell[CellGroupData[{
Cell["\<\
Notebook on the modelling of Spontaneous Parametric Down Conversion (SPDC) \
based polarization entangled source and  Heralded Multiplexed Cascaded Source \
(HMCS)\
\>", "Title",
 CellChangeTimes->{{3.892286460719638*^9, 3.892286474767694*^9}, {
  3.918035085616925*^9, 
  3.918035148592356*^9}},ExpressionUUID->"cb50704f-3a2b-45c1-8acd-\
20138b4efdd7"],

Cell[TextData[{
 "Author: Prajit Dhara (Spring 2024)\n",
 StyleBox["README",
  FontWeight->"Bold"],
 "\nThe details and performance analysis of the source is given in ",
 StyleBox["P. Dhara, S. J. Johnson, C. N. Gagatsos, P. G. Kwiat, S. Guha, \
Heralded Multiplexed High-Efficiency Cascaded Source of Dual-Rail Entangled \
Photon Pairs Using Spontaneous Parametric Down-Conversion. Phys. Rev. Appl. \
17, 034071 (2022).\n",
  FontSlant->"Italic"],
 "The extension of this source to frequency multiplexed format is done in ",
 StyleBox["Chen, K. C., Dhara, P., Heuck, M., Lee, Y., Dai, W., Guha, S., & \
Englund, D. (2022). Zero-Added-Loss Entangled Photon Multiplexing for \
Ground-and Space-Based Quantum Networks. arXiv preprint arXiv:2206.03670. \
(Look for Phys. Rev. Appl. version for more details)\n\n",
  FontSlant->"Italic"],
 StyleBox["This notebook needs the ToMatlab.m package to convert expressions \
to the Matlab format. For conversion to python format perform the following \
changes\n",
  FontWeight->"Bold",
  FontSlant->"Italic"],
 StyleBox["1) Adopt simplified variables - avoid greek symbols since for e.g. \
\[Eta] shows up as [Eta]\n2) Replace the element wise matrix multiplication \
operator .* with *\n3) Replace the element wise power operator .^ with **\n4) \
Replace any sqrt(-1) with 1j\n5) Replace exp(...) with np.exp(...)\n6) \
Replace pi with np.pi\n6) Replace array row separators \[OpenCurlyQuote];\
\[CloseCurlyQuote] with the python ],[",
  FontWeight->"Bold"]
}], "Text",
 CellChangeTimes->{{3.8922864782104197`*^9, 3.892286538758355*^9}, {
  3.8922866525311537`*^9, 3.892286689321832*^9}, {3.892286730237414*^9, 
  3.8922867603327217`*^9}, {3.918040300563922*^9, 3.918040300810231*^9}, {
  3.9242122789419746`*^9, 3.924212455198999*^9}, {3.924212511013879*^9, 
  3.924212582771961*^9}, {3.9242126719237328`*^9, 3.924212679809073*^9}},
 Background->RGBColor[
  1, 0.925, 0.925],ExpressionUUID->"4de4db4e-57b9-4d39-bc4e-87c589bd4855"],

Cell[BoxData[
 RowBox[{"<<", "ToMatlab.m"}]], "Input",
 CellChangeTimes->{{3.924212785018442*^9, 3.924212790032105*^9}},
 CellLabel->"In[83]:=",ExpressionUUID->"bf235b78-c871-49a6-a5e2-52a4eebcbd1a"],

Cell[CellGroupData[{

Cell["Preamble ", "Chapter",
 CellChangeTimes->{{3.918035164445163*^9, 
  3.918035178902104*^9}},ExpressionUUID->"14d8adce-587d-402b-8b94-\
98c892f17fef"],

Cell[CellGroupData[{

Cell["Basis Vectors and Operators", "Section",
 CellChangeTimes->{{3.832074968037138*^9, 3.8320749688727703`*^9}, {
  3.8320750135925484`*^9, 3.832075027173312*^9}, {3.835371618198537*^9, 
  3.8353716193660383`*^9}},ExpressionUUID->"f17212c3-ace8-49f4-8193-\
88aafb766cd0"],

Cell["For the spin we are limited to a qubit basis", "Text",
 CellChangeTimes->{{3.822408026527211*^9, 3.822408040105888*^9}},
 Background->RGBColor[
  0.87, 0.94, 1],ExpressionUUID->"db6f0739-c15f-48af-ad90-b89575eb79e5"],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"ketS0", "=", 
    RowBox[{"{", 
     RowBox[{"1", ",", "0"}], "}"}]}], ";"}], 
  RowBox[{"(*", "UP", "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"ketS1", "=", 
    RowBox[{"{", 
     RowBox[{"0", ",", "1"}], "}"}]}], ";"}], 
  RowBox[{"(*", "DOWN", "*)"}]}]}], "Input",
 CellChangeTimes->{{3.822408047364414*^9, 3.822408062038536*^9}, {
  3.8224115076607947`*^9, 3.822411522237186*^9}},
 CellLabel->"In[36]:=",ExpressionUUID->"48971438-373a-4c62-8eee-2be550f74bab"],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"braS0", "=", 
    RowBox[{"Transpose", "[", 
     RowBox[{"{", 
      RowBox[{"1", ",", "0"}], "}"}], "]"}]}], ";"}], 
  RowBox[{"(*", "UP", "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"braS1", "=", 
    RowBox[{"Transpose", "[", 
     RowBox[{"{", 
      RowBox[{"0", ",", "1"}], "}"}], "]"}]}], ";"}], 
  RowBox[{"(*", "DOWN", "*)"}]}]}], "Input",
 CellChangeTimes->{{3.830530926107699*^9, 3.830530945013194*^9}},
 CellLabel->"In[38]:=",ExpressionUUID->"01868f09-e100-4cdd-af78-d6e3afaa06c9"],

Cell[BoxData[
 RowBox[{
  RowBox[{"IdSpin", "=", 
   RowBox[{"SparseArray", "[", 
    RowBox[{"IdentityMatrix", "[", "2", "]"}], "]"}]}], ";"}]], "Input",
 CellChangeTimes->{{3.829954557973373*^9, 3.829954574371976*^9}},
 CellLabel->"In[40]:=",ExpressionUUID->"bb462150-f922-439c-9b80-e98e0e0b0702"],

Cell[BoxData[
 RowBox[{"(*", 
  RowBox[{
   RowBox[{"spinDm", "=", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"\[Alpha]", " ", "\[Alpha]s"}], ",", 
        RowBox[{"\[Alpha]", " ", "\[Beta]s"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"\[Alpha]s", " ", "\[Beta]"}], ",", 
        RowBox[{"\[Beta]", " ", "\[Beta]s"}]}], "}"}]}], "}"}]}], ";"}], 
  "*)"}]], "Input",
 CellChangeTimes->{{3.822408071420473*^9, 3.822408103640498*^9}, {
   3.822408229987214*^9, 3.82240829127711*^9}, 3.822408359849806*^9, {
   3.8275172326019716`*^9, 3.827517237172658*^9}},
 CellLabel->"In[41]:=",ExpressionUUID->"1eaecd85-5f92-4a17-9ca4-f9330716ede2"],

Cell["\<\
Since we limit ourselves to a max of 2 Fock states i.e. qutrit basis (for \
photonic states) the vectors for the bosonic are defined as follows\
\>", "Text",
 CellChangeTimes->{{3.8223924656615887`*^9, 3.8223924945593987`*^9}, {
  3.822392697795834*^9, 3.822392708886225*^9}, {3.832075071141205*^9, 
  3.832075076539394*^9}},
 Background->RGBColor[
  0.87, 0.94, 1],ExpressionUUID->"33d9b657-2781-44f6-8b62-6adbcbe34368"],

Cell[BoxData[
 RowBox[{
  RowBox[{"ket0", "=", 
   RowBox[{"{", 
    RowBox[{"1", ",", "0", ",", "0"}], "}"}]}], ";", " ", 
  RowBox[{"ket1", "=", 
   RowBox[{"{", 
    RowBox[{"0", ",", "1", ",", "0"}], "}"}]}], ";", " ", 
  RowBox[{"ket2", "=", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "1"}], "}"}]}], ";"}]], "Input",
 CellChangeTimes->{{3.822392497374791*^9, 3.822392576940748*^9}, {
  3.8275172207022476`*^9, 3.827517221231788*^9}},
 CellLabel->"In[42]:=",ExpressionUUID->"5bae2af4-eeac-4ecf-bc3a-3cb4a6fa056f"],

Cell["The simplest operators are defined for the sake of brevity", "Text",
 CellChangeTimes->{{3.8223926237088003`*^9, 3.822392643692501*^9}},
 Background->RGBColor[
  0.87, 0.94, 1],ExpressionUUID->"161efc38-466b-410e-92a1-857f16f08bcb"],

Cell[BoxData[
 RowBox[{
  RowBox[{"U1t1", "=", 
   RowBox[{"SparseArray", "[", 
    RowBox[{"KroneckerProduct", "[", 
     RowBox[{"ket1", ",", "ket1"}], "]"}], "]"}]}], ";", 
  RowBox[{"U1t0", "=", 
   RowBox[{"SparseArray", "[", 
    RowBox[{"KroneckerProduct", "[", 
     RowBox[{"ket0", ",", "ket1"}], "]"}], "]"}]}], ";", 
  RowBox[{"U1t2", "=", 
   RowBox[{"SparseArray", "[", 
    RowBox[{"KroneckerProduct", "[", 
     RowBox[{"ket2", ",", "ket1"}], "]"}], "]"}]}], ";"}]], "Input",
 CellChangeTimes->{{3.822392646854052*^9, 3.822392684915472*^9}, {
  3.822392742084478*^9, 3.822392742281507*^9}, {3.82240976417336*^9, 
  3.822409765771591*^9}, {3.824937796641486*^9, 3.82493782497745*^9}},
 CellLabel->"In[43]:=",ExpressionUUID->"8eb3095f-1c4a-4b98-adb7-8a6c09ff8978"],

Cell[BoxData[
 RowBox[{
  RowBox[{"U0t1", "=", 
   RowBox[{"SparseArray", "[", 
    RowBox[{"KroneckerProduct", "[", 
     RowBox[{"ket1", ",", "ket0"}], "]"}], "]"}]}], ";", 
  RowBox[{"U0t0", "=", 
   RowBox[{"SparseArray", "[", 
    RowBox[{"KroneckerProduct", "[", 
     RowBox[{"ket0", ",", "ket0"}], "]"}], "]"}]}], ";", 
  RowBox[{"U0t2", "=", 
   RowBox[{"SparseArray", "[", 
    RowBox[{"KroneckerProduct", "[", 
     RowBox[{"ket2", ",", "ket0"}], "]"}], "]"}]}], ";"}]], "Input",
 CellChangeTimes->{{3.822392731677128*^9, 3.822392749331946*^9}, {
  3.82240976754653*^9, 3.82240976908405*^9}, {3.824937862720985*^9, 
  3.8249379018574133`*^9}},
 CellLabel->"In[44]:=",ExpressionUUID->"aa437d8a-5930-46ad-89c3-10c60cc6de73"],

Cell[BoxData[
 RowBox[{
  RowBox[{"U2t1", "=", 
   RowBox[{"SparseArray", "[", 
    RowBox[{"KroneckerProduct", "[", 
     RowBox[{"ket1", ",", "ket2"}], "]"}], "]"}]}], ";", 
  RowBox[{"U2t0", "=", 
   RowBox[{"SparseArray", "[", 
    RowBox[{"KroneckerProduct", "[", 
     RowBox[{"ket0", ",", "ket2"}], "]"}], "]"}]}], ";", 
  RowBox[{"U2t2", "=", 
   RowBox[{"SparseArray", "[", 
    RowBox[{"KroneckerProduct", "[", 
     RowBox[{"ket2", ",", "ket2"}], "]"}], "]"}]}], ";"}]], "Input",
 CellChangeTimes->{{3.82239275651234*^9, 3.822392773606444*^9}, {
  3.822409771418934*^9, 3.822409773074874*^9}, {3.8249379292703757`*^9, 
  3.82493795913902*^9}},
 CellLabel->"In[45]:=",ExpressionUUID->"038a0f56-4648-4bc9-aebf-05d21e7974f0"],

Cell[BoxData[
 RowBox[{
  RowBox[{"vac", "=", 
   RowBox[{"SparseArray", "[", 
    RowBox[{"KroneckerProduct", "[", 
     RowBox[{"ket0", ",", "ket0"}], "]"}], "]"}]}], ";"}]], "Input",
 CellChangeTimes->{{3.82240793526784*^9, 3.82240795389112*^9}, {
   3.824937965125524*^9, 3.8249379695739*^9}, 3.82632715507642*^9},
 CellLabel->"In[46]:=",ExpressionUUID->"92f854ae-848c-47e3-8374-0a34448a794d"],

Cell[BoxData[
 RowBox[{
  RowBox[{"IdMode", "=", 
   RowBox[{"SparseArray", "[", 
    RowBox[{"IdentityMatrix", "[", "3", "]"}], "]"}]}], ";"}]], "Input",
 CellChangeTimes->{{3.8224115788654146`*^9, 3.822411598314404*^9}, {
  3.8249379972458353`*^9, 3.824938001526801*^9}},
 CellLabel->"In[47]:=",ExpressionUUID->"7c464376-8264-4056-bdf0-3f4ec6aff398"],

Cell[" Shorthand operations for ease of readablity", "Text",
 CellChangeTimes->{{3.918035233994217*^9, 
  3.91803524904846*^9}},ExpressionUUID->"141b6aa7-187a-4bb9-9e99-\
9277c5d3bcba"],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"Kron", "[", 
    RowBox[{"a__", ",", "b__"}], "]"}], ":=", 
   RowBox[{"KroneckerProduct", "[", 
    RowBox[{"a", ",", "b"}], "]"}]}], ";"}]], "Input",
 CellChangeTimes->{{3.822411409974462*^9, 3.822411436948596*^9}},
 Background->RGBColor[1, 0.925, 0.925],
 CellLabel->"In[48]:=",ExpressionUUID->"bd5fec39-ef74-47ff-acfb-8c8dfec8cb98"],

Cell[BoxData[
 RowBox[{
  RowBox[{"TensorProductSeq", "[", "l__", "]"}], ":=", 
  RowBox[{"Module", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"L", ",", "temp", ",", "v"}], "}"}], ",", "\[IndentingNewLine]", 
    
    RowBox[{
     RowBox[{"L", "=", 
      RowBox[{"Length", "[", "l", "]"}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"temp", "=", 
      RowBox[{"Flatten", "[", 
       RowBox[{"TensorProduct", "[", 
        RowBox[{
         RowBox[{"l", "[", 
          RowBox[{"[", 
           RowBox[{"1", ",", "All"}], "]"}], "]"}], ",", 
         RowBox[{"l", "[", 
          RowBox[{"[", 
           RowBox[{"2", ",", "All"}], "]"}], "]"}]}], "]"}], "]"}]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"Do", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"temp", "=", 
         RowBox[{"Flatten", "[", 
          RowBox[{"TensorProduct", "[", 
           RowBox[{"temp", ",", 
            RowBox[{"l", "[", 
             RowBox[{"[", 
              RowBox[{"i", ",", "All"}], "]"}], "]"}]}], "]"}], "]"}]}], 
        ";"}], ",", 
       RowBox[{"{", 
        RowBox[{"i", ",", "3", ",", "L"}], "}"}]}], "]"}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"Return", "[", "temp", "]"}]}]}], "\[IndentingNewLine]", 
   "]"}]}]], "Input",
 CellChangeTimes->{{3.82632761499316*^9, 3.826327751194935*^9}, {
  3.826327803227144*^9, 3.826327823212414*^9}, {3.826327868494289*^9, 
  3.826327955821402*^9}},
 Background->RGBColor[1, 0.925, 0.925],
 CellLabel->"In[49]:=",ExpressionUUID->"095cd585-8755-4647-b5c7-e3c2a8eaf759"]
}, Closed]],

Cell[CellGroupData[{

Cell["Fock Basis Operations", "Section",
 CellChangeTimes->{{3.833379493930099*^9, 3.833379501386911*^9}, {
  3.892387368149708*^9, 3.8923873695594463`*^9}, {3.918035264281109*^9, 
  3.918035268204588*^9}, {3.924212173381791*^9, 
  3.924212175459152*^9}},ExpressionUUID->"31d7f278-dfda-4c19-8d84-\
52aaf36b8f39"],

Cell[CellGroupData[{

Cell["Beamsplitters", "Subsection",
 CellChangeTimes->{{3.918036254057916*^9, 3.9180362629235373`*^9}, {
  3.9180387358258677`*^9, 
  3.918038740010424*^9}},ExpressionUUID->"91b59b93-b46e-486d-9c20-\
4663a55d7a4f"],

Cell[TextData[StyleBox["Output of two modes carrying n1 and n2 photons \
respectively  interfered on (\[Theta], \[Beta]) beam splitter",
 FontWeight->"Bold"]], "Text",
 CellChangeTimes->{{3.918035278297819*^9, 3.918035339945257*^9}},
 Background->RGBColor[
  0.87, 0.94, 1],ExpressionUUID->"a5f1009b-b10c-4490-b854-515b27446531"],

Cell[BoxData[
 RowBox[{
  RowBox[{"FockBSReturn", "[", 
   RowBox[{"n1_", ",", "n2_", ",", "\[Theta]_", ",", "\[Beta]_"}], "]"}], ":=", 
  RowBox[{"ArrayReshape", "[", 
   RowBox[{
    RowBox[{"Table", "[", " ", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"n1", "-", "k", "+", "l"}], ",", 
        RowBox[{"n2", "-", "l", "+", "k"}], ",", 
        RowBox[{
         FractionBox[
          SqrtBox[
           RowBox[{
            RowBox[{
             RowBox[{"(", 
              RowBox[{"n1", "-", "k", "+", "l"}], ")"}], "!"}], 
            RowBox[{
             RowBox[{"(", 
              RowBox[{"n2", "-", "l", "+", "k"}], ")"}], "!"}]}]], 
          SqrtBox[
           RowBox[{
            RowBox[{"n1", "!"}], " ", 
            RowBox[{"n2", "!"}]}]]], "*", 
         RowBox[{"Binomial", "[", 
          RowBox[{"n1", ",", "k"}], "]"}], "*", 
         RowBox[{"Binomial", "[", 
          RowBox[{"n2", ",", "l"}], "]"}], "*", 
         SuperscriptBox[
          RowBox[{"(", 
           RowBox[{"Cos", "[", "\[Theta]", "]"}], ")"}], 
          RowBox[{"n1", "-", "k"}]], "*", 
         SuperscriptBox[
          RowBox[{"(", 
           RowBox[{
            RowBox[{"-", 
             RowBox[{"Exp", "[", 
              RowBox[{"\[ImaginaryI]", " ", "\[Beta]"}], "]"}]}], "*", 
            RowBox[{"Sin", "[", "\[Theta]", "]"}]}], ")"}], "k"], "*", 
         SuperscriptBox[
          RowBox[{"(", 
           RowBox[{"Cos", "[", "\[Theta]", "]"}], ")"}], 
          RowBox[{"n2", "-", "l"}]], "*", 
         SuperscriptBox[
          RowBox[{"(", 
           RowBox[{
            RowBox[{"Exp", "[", 
             RowBox[{
              RowBox[{"-", "\[ImaginaryI]"}], " ", "\[Beta]"}], "]"}], "*", 
            RowBox[{"Sin", "[", "\[Theta]", "]"}]}], ")"}], "l"]}]}], "}"}], 
      ",", 
      RowBox[{"{", 
       RowBox[{"k", ",", "0", ",", "n1"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"l", ",", "0", ",", "n2"}], "}"}]}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{"(", 
        RowBox[{"n1", "+", "1"}], ")"}], "*", 
       RowBox[{"(", 
        RowBox[{"n2", "+", "1"}], ")"}]}], ",", "3"}], "}"}]}], "]"}], 
  RowBox[{"(*", 
   RowBox[{"Acts", " ", "on", " ", "state", " ", "vector"}], 
   "*)"}]}]], "Input",
 CellChangeTimes->{{3.826577465976227*^9, 3.826577497594589*^9}, {
  3.826577527771288*^9, 3.826577533283697*^9}, {3.826577678424273*^9, 
  3.826577708898282*^9}, {3.832087161102841*^9, 3.832087182055061*^9}},
 CellLabel->"In[50]:=",ExpressionUUID->"7ea707f7-ba85-4bcd-8c85-16732af22afe"],

Cell[TextData[{
 StyleBox["Function to evaluate the action of a (\[Theta], \[Beta]) beam \
splitter acting on two modes of a multimode density operator. This function \
evaluates the action on a single element (of a N x N dim density operator) of \
the form |n_1,n_2,...n_M><m_1,m_2,...,m_N| and applies the  beam splitter at \
positions given by mrk1 and mrk2. \n",
  FontWeight->"Bold"],
 StyleBox["Inputs to the function are:",
  FontSlant->"Italic"],
 "\n",
 StyleBox["stVeck: State ket (here  |n_1,n_2,...n_M>) expressed as a 1xM row \
vector.\nstVecb: State bra(here  <m_1,m_2,...,m_N|) expressed as a 1xM row \
vector.\nlvec: Ordered list of state kets corresponding to the rows of the \
density matrix (list of all |n_1,n_2,...n_M>, as a N x M matrix)\nmrk1: \
Position of first mode for interference\nmrk2: Position of second mode for \
interference\n (\[Theta], \[Beta]): Standard beam splitter parameters",
  FontSlant->"Italic"]
}], "Text",
 CellChangeTimes->{{3.918035278297819*^9, 3.918035339945257*^9}, {
   3.918035404068069*^9, 3.9180358368696833`*^9}, 3.918035952521424*^9, 
   3.918036102255474*^9},
 Background->RGBColor[
  0.87, 0.94, 1],ExpressionUUID->"b1fd8761-3f65-433e-9de5-a09625902f38"],

Cell[BoxData[
 RowBox[{
  RowBox[{"MultiModeBSDMat", "[", 
   RowBox[{
   "stVeck_", ",", "stVecb_", ",", "lvec_", ",", "mrk1_", ",", "mrk2_", ",", 
    "\[Theta]_", ",", "\[Beta]_"}], "]"}], ":=", 
  RowBox[{"Module", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
     "lRetk", ",", "lRetb", ",", "DMatRet", ",", "stVTempk", ",", 
      "stVTempb"}], "}"}], ",", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"lRetk", "=", 
      RowBox[{"FockBSReturn", "[", 
       RowBox[{
        RowBox[{"stVeck", "[", 
         RowBox[{"[", "mrk1", "]"}], "]"}], ",", 
        RowBox[{"stVeck", "[", 
         RowBox[{"[", "mrk2", "]"}], "]"}], ",", "\[Theta]", ",", "\[Beta]"}],
        "]"}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"lRetb", "=", 
      RowBox[{
       RowBox[{"FockBSReturn", "[", 
        RowBox[{
         RowBox[{"stVecb", "[", 
          RowBox[{"[", "mrk1", "]"}], "]"}], ",", 
         RowBox[{"stVecb", "[", 
          RowBox[{"[", "mrk2", "]"}], "]"}], ",", "\[Theta]", ",", 
         "\[Beta]"}], "]"}], "/.", 
       RowBox[{"{", 
        RowBox[{"\[ImaginaryI]", "\[Rule]", 
         RowBox[{"-", "\[ImaginaryI]"}]}], "}"}]}]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"DMatRet", "=", 
      RowBox[{"ConstantArray", "[", 
       RowBox[{"0", ",", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"Length", "[", "lvec", "]"}], ",", 
          RowBox[{"Length", "[", "lvec", "]"}]}], "}"}]}], "]"}]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"Do", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"stVTempk", "=", "stVeck"}], ";", "\[IndentingNewLine]", 
        RowBox[{"stVTempb", "=", "stVecb"}], ";", "\[IndentingNewLine]", 
        RowBox[{
         RowBox[{"stVTempk", "[", 
          RowBox[{"[", "mrk1", "]"}], "]"}], "=", 
         RowBox[{"lRetk", "[", 
          RowBox[{"[", 
           RowBox[{"i", ",", "1"}], "]"}], "]"}]}], ";", 
        RowBox[{
         RowBox[{"stVTempk", "[", 
          RowBox[{"[", "mrk2", "]"}], "]"}], "=", 
         RowBox[{"lRetk", "[", 
          RowBox[{"[", 
           RowBox[{"i", ",", "2"}], "]"}], "]"}]}], ";", 
        "\[IndentingNewLine]", 
        RowBox[{
         RowBox[{"stVTempb", "[", 
          RowBox[{"[", "mrk1", "]"}], "]"}], "=", 
         RowBox[{"lRetb", "[", 
          RowBox[{"[", 
           RowBox[{"j", ",", "1"}], "]"}], "]"}]}], ";", 
        RowBox[{
         RowBox[{"stVTempb", "[", 
          RowBox[{"[", "mrk2", "]"}], "]"}], "=", 
         RowBox[{"lRetb", "[", 
          RowBox[{"[", 
           RowBox[{"j", ",", "2"}], "]"}], "]"}]}], ";", 
        "\[IndentingNewLine]", 
        RowBox[{
         RowBox[{"DMatRet", "[", 
          RowBox[{"[", 
           RowBox[{
            RowBox[{
             RowBox[{"Position", "[", 
              RowBox[{"lvec", ",", "stVTempk"}], "]"}], "[", 
             RowBox[{"[", 
              RowBox[{"1", ",", "1"}], "]"}], "]"}], ",", 
            RowBox[{
             RowBox[{"Position", "[", 
              RowBox[{"lvec", ",", "stVTempb"}], "]"}], "[", 
             RowBox[{"[", 
              RowBox[{"1", ",", "1"}], "]"}], "]"}]}], "]"}], "]"}], "+=", 
         RowBox[{
          RowBox[{"lRetk", "[", 
           RowBox[{"[", 
            RowBox[{"i", ",", "3"}], "]"}], "]"}], "*", 
          RowBox[{"lRetb", "[", 
           RowBox[{"[", 
            RowBox[{"j", ",", "3"}], "]"}], "]"}]}]}], ";"}], ",", 
       "\[IndentingNewLine]", 
       RowBox[{"{", 
        RowBox[{"i", ",", "1", ",", 
         RowBox[{"Length", "[", "lRetk", "]"}]}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"j", ",", "1", ",", 
         RowBox[{"Length", "[", "lRetb", "]"}]}], "}"}]}], "]"}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"Return", "[", "DMatRet", "]"}]}]}], "\[IndentingNewLine]", 
   "]"}]}]], "Input",
 CellChangeTimes->{{3.826584277673193*^9, 3.826584570054122*^9}, {
   3.826584617054681*^9, 3.826584789672928*^9}, {3.826584876837774*^9, 
   3.826584912479177*^9}, {3.826585222082388*^9, 3.826585240391344*^9}, {
   3.826585682617197*^9, 3.826585865626613*^9}, {3.832087206029038*^9, 
   3.832087287726237*^9}, {3.832087318769796*^9, 3.8320873250521727`*^9}, {
   3.832087359819469*^9, 3.832087493999462*^9}, {3.832087531782475*^9, 
   3.832087685698881*^9}, {3.832087751936003*^9, 3.832088202622451*^9}, 
   3.832088249942792*^9, {3.83208834269741*^9, 3.8320883504518967`*^9}, {
   3.832088577207796*^9, 3.8320885802429237`*^9}, {3.833380024331896*^9, 
   3.833380029674192*^9}, {3.833380073660569*^9, 3.833380075662773*^9}, {
   3.833380315355338*^9, 3.833380315747259*^9}},
 CellLabel->"In[51]:=",ExpressionUUID->"b41baffb-ed76-4b5a-a292-52a355a458ce"],

Cell[TextData[{
 StyleBox["Function to evaluate the action of a (\[Theta], \[Beta]) beam \
splitter acting on two modes of a multimode density operator. This function \
evaluates the action on the complete density matrix and applies the  beam \
splitter at positions given by mrk1 and mrk2 (by repeated calls of the \
MultiModeBSDMat. ",
  FontWeight->"Bold"],
 "\n",
 StyleBox["Inputs to the function are:\nM: Complete N x N density matrix\n\
lvec: Ordered list of state kets corresponding to the rows of the density \
matrix (list of all |n_1,n_2,...n_M>, as a N x M matrix)\nmrk1: Position of \
first mode for interference\nmrk2: Position of second mode for interference\n \
(\[Theta], \[Beta]): Standard beam splitter parameters",
  FontSlant->"Italic"]
}], "Text",
 CellChangeTimes->{{3.918035278297819*^9, 3.918035339945257*^9}, {
   3.918035404068069*^9, 3.9180358368696833`*^9}, 3.918035952521424*^9, {
   3.918036007415047*^9, 3.918036113427387*^9}},
 Background->RGBColor[
  0.87, 0.94, 1],ExpressionUUID->"7e31a995-e53a-4eab-99e4-35f707513500"],

Cell[BoxData[
 RowBox[{
  RowBox[{"DMatMMBSAggregate", "[", 
   RowBox[{
   "M__", ",", "lvec__", ",", "mrk1_", ",", "mrk2_", ",", "\[Theta]_", ",", 
    "\[Beta]_"}], "]"}], ":=", 
  RowBox[{"Module", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"DMatFinal", ",", "DMatTemp", ",", "tVal"}], "}"}], ",", 
    "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"DMatFinal", "=", 
      RowBox[{"ConstantArray", "[", 
       RowBox[{"0", ",", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"Length", "[", "lvec", "]"}], ",", 
          RowBox[{"Length", "[", "lvec", "]"}]}], "}"}]}], "]"}]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"Do", "[", "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{"If", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"M", "[", 
           RowBox[{"[", 
            RowBox[{"i", ",", "j"}], "]"}], "]"}], "=!=", "0"}], ",", 
         "\[IndentingNewLine]", 
         RowBox[{
          RowBox[{"tVal", "=", 
           RowBox[{"M", "[", 
            RowBox[{"[", 
             RowBox[{"i", ",", "j"}], "]"}], "]"}]}], ";", 
          "\[IndentingNewLine]", 
          RowBox[{"DMatTemp", "=", 
           RowBox[{"MultiModeBSDMat", "[", 
            RowBox[{
             RowBox[{"lvec", "[", 
              RowBox[{"[", "i", "]"}], "]"}], ",", 
             RowBox[{"lvec", "[", 
              RowBox[{"[", "j", "]"}], "]"}], ",", "lvec", ",", "mrk1", ",", 
             "mrk2", ",", "\[Theta]", ",", "\[Beta]"}], "]"}]}], ";", 
          "\[IndentingNewLine]", 
          RowBox[{"DMatFinal", "+=", 
           RowBox[{"tVal", "*", "DMatTemp"}]}], ";"}]}], " ", "]"}], 
       "\[IndentingNewLine]", ",", 
       RowBox[{"{", 
        RowBox[{"i", ",", "1", ",", 
         RowBox[{"Length", "[", "lvec", "]"}]}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"j", ",", "1", ",", 
         RowBox[{"Length", "[", "lvec", "]"}]}], "}"}]}], "]"}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"Return", "[", "DMatFinal", "]"}]}]}], "\[IndentingNewLine]", 
   "]"}]}]], "Input",
 CellChangeTimes->{{3.826584277673193*^9, 3.826584570054122*^9}, {
   3.826584617054681*^9, 3.826584789672928*^9}, {3.826584876837774*^9, 
   3.826584912479177*^9}, {3.826585222082388*^9, 3.826585240391344*^9}, {
   3.826585682617197*^9, 3.826585865626613*^9}, {3.832087206029038*^9, 
   3.832087287726237*^9}, {3.832087318769796*^9, 3.8320873250521727`*^9}, {
   3.832087359819469*^9, 3.832087493999462*^9}, {3.832087531782475*^9, 
   3.832087685698881*^9}, {3.832087751936003*^9, 3.832088202622451*^9}, {
   3.832088249942792*^9, 3.832088260270042*^9}, 3.832088478822256*^9},
 CellLabel->"In[52]:=",ExpressionUUID->"da95d9ea-987b-4fd6-bedb-470a8bc7051d"],

Cell["\<\
Same functionality as the above two functions but optimized for sparse \
matrices\
\>", "Text",
 CellChangeTimes->{{3.918044105118408*^9, 3.9180441237260222`*^9}},
 Background->RGBColor[
  0.87, 0.94, 1],ExpressionUUID->"8f0f4971-1a22-45c1-a753-e828597bd194"],

Cell[BoxData[
 RowBox[{
  RowBox[{"MultiModeBSDMatSp", "[", 
   RowBox[{
   "stVeck_", ",", "stVecb_", ",", "lvec_", ",", "mrk1_", ",", "mrk2_", ",", 
    "\[Theta]_", ",", "\[Beta]_"}], "]"}], ":=", 
  RowBox[{"Module", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
     "lRetk", ",", "lRetb", ",", "DMatRet", ",", "stVTempk", ",", "stVTempb", 
      ",", "i1", ",", "i2"}], "}"}], ",", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"lRetk", "=", 
      RowBox[{"FockBSReturn", "[", 
       RowBox[{
        RowBox[{"stVeck", "[", 
         RowBox[{"[", "mrk1", "]"}], "]"}], ",", 
        RowBox[{"stVeck", "[", 
         RowBox[{"[", "mrk2", "]"}], "]"}], ",", "\[Theta]", ",", "\[Beta]"}],
        "]"}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"lRetb", "=", 
      RowBox[{
       RowBox[{"FockBSReturn", "[", 
        RowBox[{
         RowBox[{"stVecb", "[", 
          RowBox[{"[", "mrk1", "]"}], "]"}], ",", 
         RowBox[{"stVecb", "[", 
          RowBox[{"[", "mrk2", "]"}], "]"}], ",", "\[Theta]", ",", 
         "\[Beta]"}], "]"}], "/.", 
       RowBox[{"{", 
        RowBox[{"\[ImaginaryI]", "\[Rule]", 
         RowBox[{"-", "\[ImaginaryI]"}]}], "}"}]}]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"DMatRet", "=", 
      RowBox[{"SparseArray", "[", 
       RowBox[{"ConstantArray", "[", 
        RowBox[{"0", ",", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{"Length", "[", "lvec", "]"}], ",", 
           RowBox[{"Length", "[", "lvec", "]"}]}], "}"}]}], "]"}], "]"}]}], 
     ";", "\[IndentingNewLine]", 
     RowBox[{"Do", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"stVTempk", "=", "stVeck"}], ";", "\[IndentingNewLine]", 
        RowBox[{"stVTempb", "=", "stVecb"}], ";", "\[IndentingNewLine]", 
        RowBox[{
         RowBox[{"stVTempk", "[", 
          RowBox[{"[", "mrk1", "]"}], "]"}], "=", 
         RowBox[{"lRetk", "[", 
          RowBox[{"[", 
           RowBox[{"i", ",", "1"}], "]"}], "]"}]}], ";", 
        RowBox[{
         RowBox[{"stVTempk", "[", 
          RowBox[{"[", "mrk2", "]"}], "]"}], "=", 
         RowBox[{"lRetk", "[", 
          RowBox[{"[", 
           RowBox[{"i", ",", "2"}], "]"}], "]"}]}], ";", 
        "\[IndentingNewLine]", 
        RowBox[{
         RowBox[{"stVTempb", "[", 
          RowBox[{"[", "mrk1", "]"}], "]"}], "=", 
         RowBox[{"lRetb", "[", 
          RowBox[{"[", 
           RowBox[{"j", ",", "1"}], "]"}], "]"}]}], ";", 
        RowBox[{
         RowBox[{"stVTempb", "[", 
          RowBox[{"[", "mrk2", "]"}], "]"}], "=", 
         RowBox[{"lRetb", "[", 
          RowBox[{"[", 
           RowBox[{"j", ",", "2"}], "]"}], "]"}]}], ";", 
        "\[IndentingNewLine]", 
        RowBox[{"If", "[", 
         RowBox[{
          RowBox[{
           RowBox[{"MemberQ", "[", 
            RowBox[{"lvec", ",", "stVTempk"}], "]"}], "&&", 
           RowBox[{"MemberQ", "[", 
            RowBox[{"lvec", ",", "stVTempb"}], "]"}]}], ",", 
          "\[IndentingNewLine]", 
          RowBox[{
           RowBox[{"i1", "=", 
            RowBox[{
             RowBox[{"Position", "[", 
              RowBox[{"lvec", ",", "stVTempk"}], "]"}], "[", 
             RowBox[{"[", "1", "]"}], "]"}]}], ";", "\[IndentingNewLine]", 
           RowBox[{"i2", "=", 
            RowBox[{
             RowBox[{"Position", "[", 
              RowBox[{"lvec", ",", "stVTempb"}], "]"}], "[", 
             RowBox[{"[", "1", "]"}], "]"}]}], ";", "\[IndentingNewLine]", 
           RowBox[{
            RowBox[{"DMatRet", "[", 
             RowBox[{"[", 
              RowBox[{"i1", ",", "i2"}], "]"}], "]"}], "+=", 
            RowBox[{
             RowBox[{"lRetk", "[", 
              RowBox[{"[", 
               RowBox[{"i", ",", "3"}], "]"}], "]"}], "*", 
             RowBox[{"lRetb", "[", 
              RowBox[{"[", 
               RowBox[{"j", ",", "3"}], "]"}], "]"}]}]}]}]}], "]"}], ";"}], 
       ",", "\[IndentingNewLine]", 
       RowBox[{"{", 
        RowBox[{"i", ",", "1", ",", 
         RowBox[{"Length", "[", "lRetk", "]"}]}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"j", ",", "1", ",", 
         RowBox[{"Length", "[", "lRetb", "]"}]}], "}"}]}], "]"}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"Return", "[", "DMatRet", "]"}]}]}], "\[IndentingNewLine]", 
   "]"}]}]], "Input",
 CellChangeTimes->{{3.826584277673193*^9, 3.826584570054122*^9}, {
   3.826584617054681*^9, 3.826584789672928*^9}, {3.826584876837774*^9, 
   3.826584912479177*^9}, {3.826585222082388*^9, 3.826585240391344*^9}, {
   3.826585682617197*^9, 3.826585865626613*^9}, {3.832087206029038*^9, 
   3.832087287726237*^9}, {3.832087318769796*^9, 3.8320873250521727`*^9}, {
   3.832087359819469*^9, 3.832087493999462*^9}, {3.832087531782475*^9, 
   3.832087685698881*^9}, {3.832087751936003*^9, 3.832088202622451*^9}, 
   3.832088249942792*^9, {3.83208834269741*^9, 3.8320883504518967`*^9}, {
   3.832088577207796*^9, 3.8320885802429237`*^9}, {3.835358191453753*^9, 
   3.835358201522484*^9}, {3.8353584322837467`*^9, 3.835358442996162*^9}, {
   3.835358625324581*^9, 3.8353586724710827`*^9}, {3.835362105395396*^9, 
   3.835362149777658*^9}, {3.835362272901041*^9, 3.835362273359366*^9}, 
   3.892388982962819*^9, {3.8923890887591267`*^9, 3.892389141916873*^9}, {
   3.892389344466853*^9, 3.892389352883666*^9}, {3.892389445090437*^9, 
   3.892389446993435*^9}, {3.892389631861658*^9, 3.8923897153586187`*^9}, {
   3.892390265593033*^9, 3.892390319165544*^9}, {3.892390402867311*^9, 
   3.892390403671668*^9}},
 CellLabel->"In[53]:=",ExpressionUUID->"48b7b6ca-717e-4fa5-b325-980225b0714b"],

Cell[BoxData[
 RowBox[{
  RowBox[{"DMatMMBSAggregateSp", "[", 
   RowBox[{
   "M__", ",", "lvec__", ",", "mrk1_", ",", "mrk2_", ",", "\[Theta]_", ",", 
    "\[Beta]_"}], "]"}], ":=", 
  RowBox[{"Module", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
     "DMatFinal", ",", "DMatTemp", ",", "tVal", ",", "NZList", ",", "i", ",", 
      "j", ",", "Llvec"}], "}"}], ",", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"Llvec", "=", 
      RowBox[{"Length", "[", "lvec", "]"}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"DMatFinal", "=", 
      RowBox[{"SparseArray", "[", 
       RowBox[{"ConstantArray", "[", 
        RowBox[{"0", ",", 
         RowBox[{"{", 
          RowBox[{"Llvec", ",", "Llvec"}], "}"}]}], "]"}], "]"}]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"NZList", "=", 
      RowBox[{
       RowBox[{"SparseArray", "[", "M", "]"}], "[", 
       "\"\<NonzeroPositions\>\"", "]"}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"Print", "[", 
      RowBox[{"Length", "[", "NZList", "]"}], "]"}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"Monitor", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"Do", "[", "\[IndentingNewLine]", 
         RowBox[{
          RowBox[{
           RowBox[{"i", "=", 
            RowBox[{"NZList", "[", 
             RowBox[{"[", 
              RowBox[{"k", ",", "1"}], "]"}], "]"}]}], ";", 
           RowBox[{"j", "=", 
            RowBox[{"NZList", "[", 
             RowBox[{"[", 
              RowBox[{"k", ",", "2"}], "]"}], "]"}]}], ";", 
           "\[IndentingNewLine]", 
           RowBox[{"tVal", "=", 
            RowBox[{"M", "[", 
             RowBox[{"[", 
              RowBox[{"i", ",", "j"}], "]"}], "]"}]}], ";", 
           "\[IndentingNewLine]", 
           RowBox[{"DMatTemp", "=", 
            RowBox[{"MultiModeBSDMatSp", "[", 
             RowBox[{
              RowBox[{"lvec", "[", 
               RowBox[{"[", "i", "]"}], "]"}], ",", 
              RowBox[{"lvec", "[", 
               RowBox[{"[", "j", "]"}], "]"}], ",", "lvec", ",", "mrk1", ",", 
              "mrk2", ",", "\[Theta]", ",", "\[Beta]"}], "]"}]}], ";", 
           "\[IndentingNewLine]", 
           RowBox[{"DMatFinal", "+=", 
            RowBox[{"tVal", "*", "DMatTemp"}]}], ";"}], " ", 
          "\[IndentingNewLine]", ",", 
          RowBox[{"{", 
           RowBox[{"k", ",", "1", ",", 
            RowBox[{"Length", "[", "NZList", "]"}]}], "}"}]}], "]"}], ";"}], 
       ",", 
       RowBox[{"{", "k", "}"}]}], "]"}], ";", "\[IndentingNewLine]", 
     RowBox[{"Return", "[", "DMatFinal", "]"}]}]}], "\[IndentingNewLine]", 
   "]"}]}]], "Input",
 CellChangeTimes->{{3.826584277673193*^9, 3.826584570054122*^9}, {
   3.826584617054681*^9, 3.826584789672928*^9}, {3.826584876837774*^9, 
   3.826584912479177*^9}, {3.826585222082388*^9, 3.826585240391344*^9}, {
   3.826585682617197*^9, 3.826585865626613*^9}, {3.832087206029038*^9, 
   3.832087287726237*^9}, {3.832087318769796*^9, 3.8320873250521727`*^9}, {
   3.832087359819469*^9, 3.832087493999462*^9}, {3.832087531782475*^9, 
   3.832087685698881*^9}, {3.832087751936003*^9, 3.832088202622451*^9}, {
   3.832088249942792*^9, 3.832088260270042*^9}, 3.832088478822256*^9, {
   3.8353556477100277`*^9, 3.835355664773849*^9}, 3.835355717884877*^9, {
   3.835355758166822*^9, 3.835355776555524*^9}, {3.8353558275415*^9, 
   3.835355959177766*^9}, {3.835356033126385*^9, 3.835356040925194*^9}, {
   3.835358153103798*^9, 3.835358154187656*^9}, {3.835358371840932*^9, 
   3.835358375660544*^9}, {3.835358678251102*^9, 3.835358718444463*^9}, {
   3.835358762531184*^9, 3.835358772005468*^9}, {3.835358815607767*^9, 
   3.835358817049179*^9}, {3.835358873312982*^9, 3.835358881326173*^9}, {
   3.835358985906313*^9, 3.835358996154527*^9}, {3.835359048900223*^9, 
   3.835359050383319*^9}, 3.8923887093613853`*^9, {3.892388759270965*^9, 
   3.892388759472116*^9}, {3.8923888544376287`*^9, 3.892388872234478*^9}, {
   3.892389359702723*^9, 3.892389359925057*^9}, {3.892389514789975*^9, 
   3.892389515158589*^9}, {3.892390519655706*^9, 3.89239052875937*^9}},
 CellLabel->"In[54]:=",ExpressionUUID->"e2880865-33ff-4665-a1a3-b41a09672ebe"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Pure Loss", "Subsection",
 CellChangeTimes->{{3.9180362663783293`*^9, 
  3.9180362748147297`*^9}},ExpressionUUID->"836223b7-7bd2-456a-8ea0-\
760432d91a63"],

Cell["\<\
Returns element |n-k,k> corresponding to the action of a pure loss channel of \
transmissivity \[Eta]. 
Inputs
n: Input Fock number
k: Number of photons lost to environment
\[Eta]: Transmissivity\
\>", "Text",
 CellChangeTimes->{{3.918055899431411*^9, 3.9180559406373158`*^9}, {
  3.918055998817767*^9, 3.91805603515443*^9}},
 Background->RGBColor[
  0.87, 0.94, 1],ExpressionUUID->"500c17ab-d9db-431b-9c3b-77d8c13d82a1"],

Cell[BoxData[
 RowBox[{
  RowBox[{"LossRet", "[", 
   RowBox[{"n_", ",", "k_", ",", "\[Eta]_"}], "]"}], ":=", 
  RowBox[{
   RowBox[{"(", 
    RowBox[{
     SqrtBox[
      SuperscriptBox["\[Eta]", 
       RowBox[{"n", "-", "k"}]]], "*", 
     SuperscriptBox[
      RowBox[{"(", 
       SqrtBox[
        RowBox[{"1", "-", "\[Eta]"}]], ")"}], "k"]}], ")"}], "*", 
   FractionBox[
    SqrtBox[
     RowBox[{
      RowBox[{"Factorial", "[", "n", "]"}], " "}]], 
    SqrtBox[
     RowBox[{
      RowBox[{"Factorial", "[", 
       RowBox[{"n", "-", "k"}], "]"}], 
      RowBox[{"Factorial", "[", "k", "]"}]}]]]}]}]], "Input",
 CellChangeTimes->{{3.918036983257719*^9, 3.9180369987017193`*^9}, {
  3.918037303253186*^9, 3.918037343615987*^9}, {3.918037526572193*^9, 
  3.918037538923367*^9}, {3.918037628025799*^9, 3.918037657210659*^9}, {
  3.918037786568962*^9, 3.91803780057388*^9}},
 CellLabel->"In[55]:=",ExpressionUUID->"2f30392d-792e-4864-b1c9-e1d619e45772"],

Cell[TextData[{
 "Returns how the element |n><m| evolves under the action of a pure loss \
channel of transmissivity \[Eta].\n",
 StyleBox["Inputs",
  FontWeight->"Bold"],
 "\nn: Input Fock number of ket\nm: Input Fock number of bra\n\[Eta]: \
Transmissivity\n",
 StyleBox["Return\n",
  FontWeight->"Bold"],
 "List of all |k><l| elements arising from the action of the channel on the \
element structured as {k, l, coefficient}"
}], "Text",
 CellChangeTimes->{{3.918055899431411*^9, 3.918055997157899*^9}, {
  3.918056040263678*^9, 3.9180561298063593`*^9}},
 Background->RGBColor[
  0.87, 0.94, 1],ExpressionUUID->"74ff8bcc-e080-4828-a529-211a15892aef"],

Cell[BoxData[
 RowBox[{
  RowBox[{"LossFockElem", "[", 
   RowBox[{"n_", ",", "m_", ",", "\[Eta]_"}], "]"}], ":=", 
  RowBox[{"Module", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"minInd", ",", "ElemList", ",", "k"}], "}"}], ",", 
    "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"minInd", "=", 
      RowBox[{"Min", "[", 
       RowBox[{"{", 
        RowBox[{"n", ",", "m"}], "}"}], "]"}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"ElemList", "=", 
      RowBox[{"{", "}"}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"Table", " ", "[", 
      RowBox[{
       RowBox[{"AppendTo", "[", 
        RowBox[{"ElemList", ",", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{"n", "-", "k"}], ",", 
           RowBox[{"m", "-", "k"}], ",", 
           RowBox[{
            RowBox[{"LossRet", "[", 
             RowBox[{"n", ",", "k", ",", "\[Eta]"}], "]"}], "*", 
            RowBox[{"LossRet", "[", 
             RowBox[{"m", ",", "k", ",", "\[Eta]"}], "]"}]}]}], "}"}]}], 
        "]"}], ",", 
       RowBox[{"{", 
        RowBox[{"k", ",", "0", ",", "minInd"}], "}"}]}], "]"}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"Return", "[", "ElemList", "]"}]}]}], "\[IndentingNewLine]", 
   "]"}]}]], "Input",
 CellChangeTimes->{{3.91803629560592*^9, 3.918036323001289*^9}, {
  3.9180369030287848`*^9, 3.918036973604698*^9}, {3.91803785927722*^9, 
  3.918038043841619*^9}, {3.918038092127438*^9, 3.918038129600362*^9}},
 CellLabel->"In[56]:=",ExpressionUUID->"bda1dfbb-dcff-41c1-9af6-d71407e2b2f0"],

Cell[TextData[{
 "Returns the action of a pure loss channel of transmissivity \[Eta] on a \
density matrix at a given position\n",
 StyleBox["Inputs",
  FontWeight->"Bold"],
 "\nM: Input density matrix\nlvec: List of basis vectors \nmrk: Marker for \
position\n\[Eta]: Transmissivity\n",
 StyleBox["Return\n",
  FontWeight->"Bold"],
 "Final density matrix after action of Loss"
}], "Text",
 CellChangeTimes->{{3.918055899431411*^9, 3.918055997157899*^9}, {
  3.918056040263678*^9, 3.918056206017963*^9}},
 Background->RGBColor[
  0.87, 0.94, 1],ExpressionUUID->"449106d1-de1e-4cdb-86f7-c873c923fe3a"],

Cell[BoxData[
 RowBox[{
  RowBox[{"LossDMat", "[", 
   RowBox[{"M__", ",", "lvec__", ",", "mrk_", ",", "\[Eta]_"}], "]"}], ":=", 
  RowBox[{"Module", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
     "DMatFinal", ",", "tVal", ",", "nv", ",", "mv", ",", "rowVec", ",", 
      "colVec", ",", "RetList", ",", "rowPos", ",", "colPos"}], "}"}], ",", 
    "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"DMatFinal", "=", 
      RowBox[{"ConstantArray", "[", 
       RowBox[{"0", ",", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"Length", "[", "lvec", "]"}], ",", 
          RowBox[{"Length", "[", "lvec", "]"}]}], "}"}]}], "]"}]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"Do", "[", "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{"If", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"M", "[", 
           RowBox[{"[", 
            RowBox[{"i", ",", "j"}], "]"}], "]"}], "=!=", "0"}], ",", 
         "\[IndentingNewLine]", 
         RowBox[{
          RowBox[{"tVal", "=", 
           RowBox[{"M", "[", 
            RowBox[{"[", 
             RowBox[{"i", ",", "j"}], "]"}], "]"}]}], ";", 
          "\[IndentingNewLine]", 
          RowBox[{"nv", "=", 
           RowBox[{"lvec", "[", 
            RowBox[{"[", 
             RowBox[{"i", ",", "mrk"}], "]"}], "]"}]}], ";", " ", 
          RowBox[{"mv", "=", 
           RowBox[{"lvec", "[", 
            RowBox[{"[", 
             RowBox[{"j", ",", "mrk"}], "]"}], "]"}]}], ";", 
          "\[IndentingNewLine]", 
          RowBox[{"RetList", "=", 
           RowBox[{"LossFockElem", "[", 
            RowBox[{"nv", ",", "mv", ",", "\[Eta]"}], "]"}]}], ";", 
          "\[IndentingNewLine]", 
          RowBox[{"Table", "[", "\[IndentingNewLine]", 
           RowBox[{
            RowBox[{
             RowBox[{"rowVec", "=", 
              RowBox[{"lvec", "[", 
               RowBox[{"[", "i", "]"}], "]"}]}], ";", 
             RowBox[{
              RowBox[{"rowVec", "[", 
               RowBox[{"[", "mrk", "]"}], "]"}], "=", 
              RowBox[{"RetList", "[", 
               RowBox[{"[", 
                RowBox[{"k", ",", "1"}], "]"}], "]"}]}], ";", 
             "\[IndentingNewLine]", 
             RowBox[{"colVec", "=", 
              RowBox[{"lvec", "[", 
               RowBox[{"[", "j", "]"}], "]"}]}], ";", 
             RowBox[{
              RowBox[{"colVec", "[", 
               RowBox[{"[", "mrk", "]"}], "]"}], "=", 
              RowBox[{"RetList", "[", 
               RowBox[{"[", 
                RowBox[{"k", ",", "2"}], "]"}], "]"}]}], ";", 
             "\[IndentingNewLine]", 
             RowBox[{"rowPos", "=", 
              RowBox[{
               RowBox[{"Position", "[", 
                RowBox[{"lvec", ",", "rowVec"}], "]"}], "[", 
               RowBox[{"[", "1", "]"}], "]"}]}], ";", " ", 
             RowBox[{"colPos", "=", 
              RowBox[{
               RowBox[{"Position", "[", 
                RowBox[{"lvec", ",", "colVec"}], "]"}], "[", 
               RowBox[{"[", "1", "]"}], "]"}]}], ";", "\[IndentingNewLine]", 
             RowBox[{
              RowBox[{"DMatFinal", "[", 
               RowBox[{"[", 
                RowBox[{"rowPos", ",", "colPos"}], "]"}], "]"}], "+=", 
              RowBox[{"tVal", "*", 
               RowBox[{"RetList", "[", 
                RowBox[{"[", 
                 RowBox[{"k", ",", "3"}], "]"}], "]"}]}]}], ";"}], 
            "\[IndentingNewLine]", ",", 
            RowBox[{"{", 
             RowBox[{"k", ",", "1", ",", 
              RowBox[{"Length", "[", "RetList", "]"}]}], "}"}]}], "]"}], 
          ";"}]}], "]"}], "\[IndentingNewLine]", ",", 
       RowBox[{"{", 
        RowBox[{"i", ",", "1", ",", 
         RowBox[{"Length", "[", "lvec", "]"}]}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"j", ",", "1", ",", 
         RowBox[{"Length", "[", "lvec", "]"}]}], "}"}]}], "]"}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"Return", "[", "DMatFinal", "]"}]}]}], "\[IndentingNewLine]", 
   "]"}]}]], "Input",
 CellChangeTimes->{{3.9180361788620653`*^9, 3.918036233267662*^9}, {
  3.918038132662517*^9, 3.918038467958146*^9}, {3.918038504779563*^9, 
  3.918038558557783*^9}, {3.9180385926900787`*^9, 3.918038669238365*^9}},
 CellLabel->"In[57]:=",ExpressionUUID->"595ac4e2-67d2-4596-adbe-ad4dbcd4210c"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Testing", "Subsection",
 CellChangeTimes->{{3.918038883943338*^9, 
  3.918038891890681*^9}},ExpressionUUID->"f74dba2f-1e96-4cd6-be98-\
981969e69e01"],

Cell["Testing single mode loss", "Text",
 CellChangeTimes->{{3.918039040269535*^9, 3.918039049374282*^9}},
 Background->RGBColor[
  0.87, 0.94, 1],ExpressionUUID->"43168903-8a53-49a3-8d52-e3b4b9412f3a"],

Cell[BoxData[{
 RowBox[{
  RowBox[{"DMTest", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"0", ",", "0", ",", "0"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "0", ",", "0"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "0", ",", "1"}], "}"}]}], "}"}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"VecTest", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", "0", "}"}], ",", 
     RowBox[{"{", "1", "}"}], ",", 
     RowBox[{"{", "2", "}"}]}], "}"}]}], ";"}]}], "Input",
 CellChangeTimes->{{3.91803905103728*^9, 3.918039101565347*^9}},
 CellLabel->"In[58]:=",ExpressionUUID->"39183672-4e9a-4d10-b919-6ef86aafc324"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"LossDMat", "[", 
   RowBox[{"DMTest", ",", "VecTest", ",", "1", ",", "\[Eta]"}], "]"}], "//", 
  "MatrixForm"}]], "Input",
 CellChangeTimes->{{3.918039092247727*^9, 3.9180391314462137`*^9}, {
  3.918039225797084*^9, 3.918039230331677*^9}},
 CellLabel->"In[60]:=",ExpressionUUID->"c19aab7b-bf06-4236-bff5-bd7496fc25e9"],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {
      SuperscriptBox[
       RowBox[{"(", 
        RowBox[{"1", "-", "\[Eta]"}], ")"}], "2"], "0", "0"},
     {"0", 
      RowBox[{"2", " ", 
       RowBox[{"(", 
        RowBox[{"1", "-", "\[Eta]"}], ")"}], " ", "\[Eta]"}], "0"},
     {"0", "0", 
      SuperscriptBox["\[Eta]", "2"]}
    },
    GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}}], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Output",
 CellChangeTimes->{
  3.918039132138364*^9, 3.918039230683103*^9, 3.91803934915867*^9, 
   3.91803981214826*^9, {3.924261692113085*^9, 3.924261692430585*^9}},
 CellLabel->
  "Out[60]//MatrixForm=",ExpressionUUID->"e41ebb7c-416e-4263-bcfe-\
630a2d21140c"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"DMTest", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "0", ",", "0"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "0", ",", "0"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "0", ",", "0"}], "}"}]}], "}"}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"LossDMat", "[", 
    RowBox[{"DMTest", ",", "VecTest", ",", "1", ",", "\[Eta]"}], "]"}], "//", 
   "FullSimplify"}], "//", "MatrixForm"}]}], "Input",
 CellChangeTimes->{{3.918039765653825*^9, 3.918039770660593*^9}},
 CellLabel->"In[61]:=",ExpressionUUID->"a6153760-081e-403b-8e86-459bf34c34df"],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {"1", "0", "0"},
     {"0", "0", "0"},
     {"0", "0", "0"}
    },
    GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}}], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Output",
 CellChangeTimes->{
  3.918039771214061*^9, 3.9180398121539392`*^9, {3.924261692136707*^9, 
   3.92426169244585*^9}},
 CellLabel->
  "Out[62]//MatrixForm=",ExpressionUUID->"19c5d0b5-c956-4fd5-ad53-\
4e3dc505bf69"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"DMTest", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"1", "/", "2"}], ",", "0", ",", 
       RowBox[{"1", "/", "2"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "0", ",", "0"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"1", "/", "2"}], ",", "0", ",", 
       RowBox[{"1", "/", "2"}]}], "}"}]}], "}"}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"LossDMat", "[", 
    RowBox[{"DMTest", ",", "VecTest", ",", "1", ",", "\[Eta]"}], "]"}], "//", 
   "FullSimplify"}], "//", "MatrixForm"}]}], "Input",
 CellChangeTimes->{{3.918039147549237*^9, 3.918039223020104*^9}},
 CellLabel->"In[63]:=",ExpressionUUID->"fd4f4e7c-7bb0-407c-b6de-2c392a3ee755"],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {
      RowBox[{"1", "+", 
       RowBox[{
        FractionBox["1", "2"], " ", 
        RowBox[{"(", 
         RowBox[{
          RowBox[{"-", "2"}], "+", "\[Eta]"}], ")"}], " ", "\[Eta]"}]}], "0", 
      
      FractionBox[
       SqrtBox[
        SuperscriptBox["\[Eta]", "2"]], "2"]},
     {"0", 
      RowBox[{"-", 
       RowBox[{"(", 
        RowBox[{
         RowBox[{"(", 
          RowBox[{
           RowBox[{"-", "1"}], "+", "\[Eta]"}], ")"}], " ", "\[Eta]"}], 
        ")"}]}], "0"},
     {
      FractionBox[
       SqrtBox[
        SuperscriptBox["\[Eta]", "2"]], "2"], "0", 
      FractionBox[
       SuperscriptBox["\[Eta]", "2"], "2"]}
    },
    GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}}], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Output",
 CellChangeTimes->{{3.918039163307358*^9, 3.918039223302915*^9}, 
   3.9180393491650753`*^9, 3.918039812174885*^9, {3.92426169223503*^9, 
   3.924261692452354*^9}},
 CellLabel->
  "Out[64]//MatrixForm=",ExpressionUUID->"6a7b5791-e2b8-4453-a287-\
28d15174de17"]
}, Open  ]],

Cell["Testing multimode loss", "Text",
 CellChangeTimes->{{3.918039040269535*^9, 3.918039049374282*^9}, {
  3.918039860763476*^9, 3.918039866976116*^9}},
 Background->RGBColor[
  0.87, 0.94, 1],ExpressionUUID->"431108c5-df9d-494f-a896-1546b2503780"],

Cell[BoxData[{
 RowBox[{
  RowBox[{"DMTest", "=", 
   RowBox[{"Kron", "[", 
    RowBox[{
     RowBox[{"Flatten", "[", 
      RowBox[{"KroneckerProduct", "[", 
       RowBox[{"ket2", ",", "ket0"}], "]"}], "]"}], ",", 
     RowBox[{"Flatten", "[", 
      RowBox[{"KroneckerProduct", "[", 
       RowBox[{"ket2", ",", "ket0"}], "]"}], "]"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"VecTest2", "=", 
   RowBox[{"ArrayReshape", "[", 
    RowBox[{
     RowBox[{"Table", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"k1", ",", "k2"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"k1", ",", "0", ",", "2"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"k2", ",", "0", ",", "2"}], "}"}]}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"9", ",", "2"}], "}"}]}], "]"}]}], ";"}]}], "Input",
 CellChangeTimes->{{3.9180398846946287`*^9, 3.918039960329083*^9}, {
  3.918039990930437*^9, 3.918039995804549*^9}, {3.918040046746872*^9, 
  3.918040061516951*^9}},
 CellLabel->"In[65]:=",ExpressionUUID->"ab2554b5-fc04-4bee-b42a-651551e2bf83"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"LossDMat", "[", 
   RowBox[{"DMTest", ",", "VecTest2", ",", "1", ",", "\[Eta]"}], "]"}], "//", 
  "MatrixForm"}]], "Input",
 CellChangeTimes->{3.9180400084039803`*^9},
 CellLabel->"In[67]:=",ExpressionUUID->"39368815-9a78-4dff-b0de-07b8745411f9"],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {
      SuperscriptBox[
       RowBox[{"(", 
        RowBox[{"1", "-", "\[Eta]"}], ")"}], "2"], "0", "0", "0", "0", "0", 
      "0", "0", "0"},
     {"0", "0", "0", "0", "0", "0", "0", "0", "0"},
     {"0", "0", "0", "0", "0", "0", "0", "0", "0"},
     {"0", "0", "0", 
      RowBox[{"2", " ", 
       RowBox[{"(", 
        RowBox[{"1", "-", "\[Eta]"}], ")"}], " ", "\[Eta]"}], "0", "0", "0", 
      "0", "0"},
     {"0", "0", "0", "0", "0", "0", "0", "0", "0"},
     {"0", "0", "0", "0", "0", "0", "0", "0", "0"},
     {"0", "0", "0", "0", "0", "0", 
      SuperscriptBox["\[Eta]", "2"], "0", "0"},
     {"0", "0", "0", "0", "0", "0", "0", "0", "0"},
     {"0", "0", "0", "0", "0", "0", "0", "0", "0"}
    },
    GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}}], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Output",
 CellChangeTimes->{
  3.91804000896751*^9, 3.918040068205567*^9, {3.924261692257885*^9, 
   3.924261692476852*^9}},
 CellLabel->
  "Out[67]//MatrixForm=",ExpressionUUID->"1fb2b3bb-1061-47db-b0bb-\
a91ff793d766"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"DMTest", "=", 
   RowBox[{"Kron", "[", 
    RowBox[{
     RowBox[{"Flatten", "[", 
      RowBox[{"KroneckerProduct", "[", 
       RowBox[{"ket2", ",", "ket2"}], "]"}], "]"}], ",", 
     RowBox[{"Flatten", "[", 
      RowBox[{"KroneckerProduct", "[", 
       RowBox[{"ket2", ",", "ket2"}], "]"}], "]"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"LossDMat", "[", 
    RowBox[{
     RowBox[{"LossDMat", "[", 
      RowBox[{"DMTest", ",", "VecTest2", ",", "1", ",", "\[Eta]"}], "]"}], 
     ",", "VecTest2", ",", "2", ",", "\[Eta]"}], "]"}], "//", "Tr"}], "//", 
  "Simplify"}]}], "Input",
 CellChangeTimes->{{3.9180398846946287`*^9, 3.918039960329083*^9}, {
  3.918039990930437*^9, 3.918039995804549*^9}, {3.918040046746872*^9, 
  3.918040159160856*^9}},
 CellLabel->"In[68]:=",ExpressionUUID->"8d2502b4-505b-43e4-9376-8c9351f889e7"],

Cell[BoxData["1"], "Output",
 CellChangeTimes->{{3.9180400951253147`*^9, 3.918040159436586*^9}, {
  3.924261692264477*^9, 3.9242616924819536`*^9}},
 CellLabel->"Out[69]=",ExpressionUUID->"3a5addd4-cc73-480f-98f5-2a5a4274c51a"]
}, Open  ]]
}, Closed]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Spontaneous Parametric Down Conversion (SPDC) State", "Chapter",
 CellChangeTimes->{{3.91804025395381*^9, 
  3.918040262952755*^9}},ExpressionUUID->"78b09564-4fdc-40bc-b187-\
55abd708f9a1"],

Cell[CellGroupData[{

Cell["State Initialization and Loss", "Section",
 CellChangeTimes->{{3.918078669941042*^9, 
  3.918078678547233*^9}},ExpressionUUID->"ea26b07c-321f-40a9-b8e7-\
dd87cb354b10"],

Cell["Photon number distribution function", "Text",
 CellChangeTimes->{{3.924212187541802*^9, 3.92421219728984*^9}},
 Background->RGBColor[
  0.87, 0.94, 1],ExpressionUUID->"67c2e5ae-5d0f-4b86-a10d-1487efa5bf54"],

Cell[BoxData[
 RowBox[{
  RowBox[{"ProbFn", "[", "n_", "]"}], ":=", 
  RowBox[{
   RowBox[{"(", 
    RowBox[{"n", "+", "1"}], ")"}], "*", 
   FractionBox[
    SuperscriptBox["Ns", "n"], 
    SuperscriptBox[
     RowBox[{"(", 
      RowBox[{"Ns", "+", "1"}], ")"}], 
     RowBox[{"n", "+", "2"}]]]}]}]], "Input",
 CellChangeTimes->{{3.918043397325975*^9, 3.9180434574029093`*^9}},
 CellLabel->"In[70]:=",ExpressionUUID->"972dba28-8c79-4229-a144-eb7e4c772a79"],

Cell["State ket  initialization", "Text",
 CellChangeTimes->{{3.91804073540003*^9, 3.918040740193012*^9}, {
  3.918043212293078*^9, 3.918043233660079*^9}, {3.918056221498893*^9, 
  3.918056228676548*^9}},
 Background->RGBColor[
  0.87, 0.94, 1],ExpressionUUID->"211c00c8-c138-4e78-9cd5-ce342c54ce76"],

Cell[BoxData[
 RowBox[{
  RowBox[{"SPDCKet", "=", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      RowBox[{
       SqrtBox["p0"], 
       RowBox[{"Flatten", "[", 
        RowBox[{"KroneckerProduct", "[", 
         RowBox[{"ket0", ",", "ket0", ",", "ket0", ",", "ket0"}], "]"}], 
        "]"}]}], "+", 
      RowBox[{
       SqrtBox[
        RowBox[{"p1", "/", "2"}]], 
       RowBox[{"Flatten", "[", 
        RowBox[{"KroneckerProduct", "[", 
         RowBox[{"ket1", ",", "ket0", ",", "ket0", ",", "ket1"}], "]"}], 
        "]"}]}], "+", 
      RowBox[{
       SqrtBox[
        RowBox[{"p1", "/", "2"}]], 
       RowBox[{"Flatten", "[", 
        RowBox[{"KroneckerProduct", "[", 
         RowBox[{"ket0", ",", "ket1", ",", "ket1", ",", "ket0"}], "]"}], 
        "]"}]}], "+", 
      RowBox[{
       SqrtBox[
        RowBox[{"p2", "/", "3"}]], 
       RowBox[{"Flatten", "[", 
        RowBox[{"KroneckerProduct", "[", 
         RowBox[{"ket1", ",", "ket1", ",", "ket1", ",", "ket1"}], "]"}], 
        "]"}]}], "+", 
      RowBox[{
       SqrtBox[
        RowBox[{"p2", "/", "3"}]], 
       RowBox[{"Flatten", "[", 
        RowBox[{"KroneckerProduct", "[", 
         RowBox[{"ket0", ",", "ket2", ",", "ket2", ",", "ket0"}], "]"}], 
        "]"}]}], "+", 
      RowBox[{
       SqrtBox[
        RowBox[{"p2", "/", "3"}]], 
       RowBox[{"Flatten", "[", 
        RowBox[{"KroneckerProduct", "[", 
         RowBox[{"ket2", ",", "ket0", ",", "ket0", ",", "ket2"}], "]"}], 
        "]"}]}]}], ")"}], "/", 
    SqrtBox[
     RowBox[{"p0", "+", "p1", "+", "p2"}]]}]}], ";"}]], "Input",
 CellChangeTimes->{{3.918040600192806*^9, 3.918040691334069*^9}, 
   3.9180407225949917`*^9, {3.918040960506315*^9, 3.918040976464242*^9}},
 CellLabel->"In[71]:=",ExpressionUUID->"15209ca1-c1dd-46ab-a25f-ab145a7bcf23"],

Cell["State density operator  initialization", "Text",
 CellChangeTimes->{{3.91804073540003*^9, 3.918040740193012*^9}, {
  3.918043212293078*^9, 3.918043233660079*^9}, {3.918056221498893*^9, 
  3.918056228676548*^9}, {3.918056259492001*^9, 3.918056262967871*^9}},
 Background->RGBColor[
  0.87, 0.94, 1],ExpressionUUID->"2d92e80e-56f5-44ea-89a2-c0e3e48c5d8b"],

Cell[BoxData[
 RowBox[{
  RowBox[{"SPDCDMat", "=", 
   RowBox[{"Kron", "[", 
    RowBox[{"SPDCKet", ",", "SPDCKet"}], "]"}]}], ";"}]], "Input",
 CellChangeTimes->{{3.918040698352056*^9, 3.918040718825774*^9}},
 CellLabel->"In[72]:=",ExpressionUUID->"401c8d67-790c-4c97-bfd9-bc95372f9578"],

Cell["\<\
States after loss (applying general loss channels to the four separate modes)\
\
\>", "Text",
 CellChangeTimes->{{3.91804073540003*^9, 3.918040740193012*^9}, {
  3.918043212293078*^9, 3.918043233660079*^9}},
 Background->RGBColor[
  0.87, 0.94, 1],ExpressionUUID->"d1e1f06d-48ff-475f-85a2-feddcc1a2ae3"],

Cell[BoxData[
 RowBox[{
  RowBox[{"VecSPDC", "=", 
   RowBox[{"ArrayReshape", "[", 
    RowBox[{
     RowBox[{"Table", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"k1", ",", "k2", ",", "k3", ",", "k4"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"k1", ",", "0", ",", "2"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"k2", ",", "0", ",", "2"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"k3", ",", "0", ",", "2"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"k4", ",", "0", ",", "2"}], "}"}]}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"81", ",", "4"}], "}"}]}], "]"}]}], ";"}]], "Input",
 CellChangeTimes->{{3.91804080640174*^9, 3.918040826145371*^9}},
 CellLabel->"In[73]:=",ExpressionUUID->"ac1da23b-0ae7-4cd0-b51c-ee30a6a9600d"],

Cell[BoxData[{
 RowBox[{
  RowBox[{"SPDCInit", "=", "SPDCDMat"}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"SPDC1", "=", 
   RowBox[{"LossDMat", "[", 
    RowBox[{"SPDCDMat", ",", "VecSPDC", ",", "1", ",", "\[Eta]1"}], "]"}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"SPDC2", "=", 
   RowBox[{"LossDMat", "[", 
    RowBox[{"SPDC1", ",", "VecSPDC", ",", "2", ",", "\[Eta]2"}], "]"}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"SPDC3", "=", 
   RowBox[{"LossDMat", "[", 
    RowBox[{"SPDC2", ",", "VecSPDC", ",", "3", ",", "\[Eta]3"}], "]"}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"SPDC4", "=", 
   RowBox[{"LossDMat", "[", 
    RowBox[{"SPDC3", ",", "VecSPDC", ",", "4", ",", "\[Eta]4"}], "]"}]}], 
  ";"}]}], "Input",
 CellChangeTimes->{{3.91804076764506*^9, 3.918040791584743*^9}, {
  3.918040831365759*^9, 3.91804088811418*^9}},
 CellLabel->"In[74]:=",ExpressionUUID->"ab886123-0acf-4364-9241-036550aa3b2b"],

Cell["\<\
State density matrix (assuming modes 1,2 and 3,4 have the same losses)\
\>", "Text",
 CellChangeTimes->{{3.918043241036032*^9, 3.918043268917938*^9}},
 Background->RGBColor[
  0.87, 0.94, 1],ExpressionUUID->"cd3af855-7d76-4d63-a308-31e64cd4f85b"],

Cell[BoxData[
 RowBox[{
  RowBox[{"SPDCMatrixAfterLoss", "=", 
   RowBox[{"SPDC4", "/.", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"\[Eta]1", "->", "\[Eta]A"}], ",", 
      RowBox[{"\[Eta]2", "->", "\[Eta]A"}], ",", 
      RowBox[{"\[Eta]3", "->", "\[Eta]B"}], ",", 
      RowBox[{"\[Eta]4", "->", "\[Eta]B"}], ",", 
      RowBox[{"p0", "->", 
       RowBox[{"ProbFn", "[", "0", "]"}]}], ",", 
      RowBox[{"p1", "->", 
       RowBox[{"ProbFn", "[", "1", "]"}]}], ",", 
      RowBox[{"p2", "->", 
       RowBox[{"ProbFn", "[", "2", "]"}]}]}], "}"}]}]}], ";"}]], "Input",
 CellChangeTimes->{{3.918043271530753*^9, 3.918043372779279*^9}, {
  3.918043506498136*^9, 3.918043536512993*^9}},
 CellLabel->"In[79]:=",ExpressionUUID->"2d71162a-8ba7-4ee9-b734-8d03685e1bb2"],

Cell[CellGroupData[{

Cell["File Saving", "Subsubsection",
 CellChangeTimes->{{3.924212223643115*^9, 
  3.924212233154134*^9}},ExpressionUUID->"b5fb754e-77ac-436f-a6c4-\
49c3ec9207ac"],

Cell[BoxData[
 RowBox[{
  RowBox[{"SPDCAfterLossMATLAB", "=", 
   RowBox[{"SPDC4", "/.", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"\[Eta]1", "->", "eA"}], ",", 
      RowBox[{"\[Eta]2", "->", "eA"}], ",", 
      RowBox[{"\[Eta]3", "->", "eB"}], ",", 
      RowBox[{"\[Eta]4", "->", "eB"}], ",", 
      RowBox[{"p0", "->", 
       RowBox[{"ProbFn", "[", "0", "]"}]}], ",", 
      RowBox[{"p1", "->", 
       RowBox[{"ProbFn", "[", "1", "]"}]}], ",", 
      RowBox[{"p2", "->", 
       RowBox[{"ProbFn", "[", "2", "]"}]}]}], "}"}]}]}], ";"}]], "Input",
 CellChangeTimes->{{3.924212754454142*^9, 3.924212778032202*^9}},
 CellLabel->"In[80]:=",ExpressionUUID->"63a8778b-b776-40ab-a55e-f257326fbb31"],

Cell[BoxData[
 RowBox[{"WriteMatlab", "[", 
  RowBox[{"SPDCAfterLossMATLAB", ",", " ", "\"\<dens_mat_SPDC.m\>\""}], 
  "]"}]], "Input",
 CellChangeTimes->{{3.924212779548608*^9, 3.924212854528118*^9}},
 CellLabel->"In[84]:=",ExpressionUUID->"654c5a02-2701-47cd-a8b8-3f0bb6ac6425"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Testing", "Subsubsection",
 CellChangeTimes->{{3.924261755712817*^9, 3.9242617575039387`*^9}, 
   3.9242621276538*^9},ExpressionUUID->"f1f53dfc-10cd-4693-b7dc-66271edad814"],

Cell["Test Case 1 : State with no losses and Ns-  = 1e-3", "Text",
 CellChangeTimes->{{3.9242617623380136`*^9, 3.924261780297464*^9}},
 Background->RGBColor[
  0.87, 0.94, 1],ExpressionUUID->"5a7a8759-980f-494b-87e6-36c3a3e1f7b0"],

Cell[BoxData[{
 RowBox[{
  RowBox[{"SPDCTest1", "=", 
   RowBox[{"SPDCAfterLossMATLAB", "/.", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"eA", "->", "1"}], ",", 
      RowBox[{"eB", "->", "1."}], ",", 
      RowBox[{"Ns", "->", 
       SuperscriptBox["10", 
        RowBox[{"-", "3"}]]}]}], "}"}]}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"Export", "[", 
   RowBox[{"\"\<SPDCTest1.txt\>\"", ",", "SPDCTest1", ",", "\"\<CSV\>\""}], 
   "]"}], ";"}]}], "Input",
 CellChangeTimes->{{3.924261784015253*^9, 3.924261865368359*^9}, {
   3.92426191657752*^9, 3.9242619257186823`*^9}, 3.924261973524811*^9},
 CellLabel->"In[93]:=",ExpressionUUID->"93c407ac-39de-4288-9f7f-2c428630cb22"],

Cell["\<\
Test Case 2 : State with 10 dB loss (\[Eta]A=\[Eta]B=0.1) and Ns-  = 1e-3\
\>", "Text",
 CellChangeTimes->{{3.9242617623380136`*^9, 3.924261780297464*^9}, {
  3.924261994341853*^9, 3.924262020487157*^9}},
 Background->RGBColor[
  0.87, 0.94, 1],ExpressionUUID->"be94e640-f9a7-44ac-8685-6570a2efacc4"],

Cell[BoxData[{
 RowBox[{
  RowBox[{"SPDCTest2", "=", 
   RowBox[{"SPDCAfterLossMATLAB", "/.", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"eA", "->", "0.1"}], ",", 
      RowBox[{"eB", "->", "0.1"}], ",", 
      RowBox[{"Ns", "->", 
       SuperscriptBox["10", 
        RowBox[{"-", "3"}]]}]}], "}"}]}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"Export", "[", 
   RowBox[{"\"\<SPDCTest2.txt\>\"", ",", "SPDCTest2", ",", "\"\<CSV\>\""}], 
   "]"}], ";"}]}], "Input",
 CellChangeTimes->{{3.924261784015253*^9, 3.924261865368359*^9}, {
  3.92426191657752*^9, 3.9242619257186823`*^9}, {3.924261973524811*^9, 
  3.924262026311095*^9}},ExpressionUUID->"73cdbb75-7fa5-44dd-869f-\
631846912996"],

Cell["\<\
Test Case 3 : State with 10 dB loss (\[Eta]A=0.1) and 20 dB \
loss(\[Eta]B=0.01) and Ns-  = 1e-3\
\>", "Text",
 CellChangeTimes->{{3.9242617623380136`*^9, 3.924261780297464*^9}, {
  3.924262036255006*^9, 3.92426207076912*^9}},
 Background->RGBColor[
  0.87, 0.94, 1],ExpressionUUID->"6558d85e-7311-4b99-9596-3651e826a2fb"],

Cell[BoxData[{
 RowBox[{
  RowBox[{"SPDCTest3", "=", 
   RowBox[{"SPDCAfterLossMATLAB", "/.", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"eA", "->", "0.1"}], ",", 
      RowBox[{"eB", "->", "0.01"}], ",", 
      RowBox[{"Ns", "->", 
       SuperscriptBox["10", 
        RowBox[{"-", "3"}]]}]}], "}"}]}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"Export", "[", 
   RowBox[{"\"\<SPDCTest3.txt\>\"", ",", "SPDCTest3", ",", "\"\<CSV\>\""}], 
   "]"}], ";"}]}], "Input",
 CellChangeTimes->{{3.924261784015253*^9, 3.924261865368359*^9}, {
   3.92426191657752*^9, 3.9242619257186823`*^9}, 3.924261973524811*^9, {
   3.924262040317253*^9, 3.9242620411261067`*^9}, {3.924262073635367*^9, 
   3.924262082863365*^9}},ExpressionUUID->"1ddbc7d7-ff4e-4522-8645-\
878c49d98aad"],

Cell["\<\
Test Case 4 : State with 20 dB loss (\[Eta]A=0.01) and 10 dB \
loss(\[Eta]B=0.1) and Ns-  = 1e-1\
\>", "Text",
 CellChangeTimes->{{3.9242617623380136`*^9, 3.924261780297464*^9}, {
  3.924261994341853*^9, 3.9242620388650084`*^9}, {3.924262093830114*^9, 
  3.924262113634525*^9}},
 Background->RGBColor[
  0.87, 0.94, 1],ExpressionUUID->"ebb49dda-e5af-4ca2-8e7d-60cc0d5d04a2"],

Cell[BoxData[{
 RowBox[{
  RowBox[{"SPDCTest4", "=", 
   RowBox[{"SPDCAfterLossMATLAB", "/.", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"eA", "->", "0.01"}], ",", 
      RowBox[{"eB", "->", "0.1"}], ",", 
      RowBox[{"Ns", "->", 
       SuperscriptBox["10", 
        RowBox[{"-", "1"}]]}]}], "}"}]}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"Export", "[", 
   RowBox[{"\"\<SPDCTest4.txt\>\"", ",", "SPDCTest4", ",", "\"\<CSV\>\""}], 
   "]"}], ";"}]}], "Input",
 CellChangeTimes->{{3.924261784015253*^9, 3.924261865368359*^9}, {
  3.92426191657752*^9, 3.9242619257186823`*^9}, {3.924261973524811*^9, 
  3.924262042594325*^9}, {3.924262085117111*^9, 
  3.924262108845314*^9}},ExpressionUUID->"43c8e794-bf9f-4713-81cf-\
ee68fa4e1333"]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Heralded Multiplexed Cascaded Source (HMCS)", "Chapter",
 CellChangeTimes->{{3.918043550968791*^9, 3.918043554446805*^9}, {
  3.918043590484662*^9, 
  3.918043613900602*^9}},ExpressionUUID->"c46e9978-8d33-47f4-ae83-\
b0915ecf9f1b"],

Cell["\<\
Considering two copies of the SPDC based polarization entanglement state\
\>", "Text",
 CellChangeTimes->{{3.918043241036032*^9, 3.918043268917938*^9}, 
   3.918056287837054*^9, {3.924212263868815*^9, 3.9242122753036757`*^9}},
 Background->RGBColor[
  0.87, 0.94, 1],ExpressionUUID->"5dcd7607-dca9-4fca-8c9a-8d7dac0fe916"],

Cell[BoxData[{
 RowBox[{
  RowBox[{"SPDCSrc1", "=", 
   RowBox[{"Simplify", "[", 
    RowBox[{"SPDC4", "/.", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"\[Eta]1", "->", "\[Eta]A"}], ",", 
       RowBox[{"\[Eta]2", "->", "\[Eta]A"}], ",", 
       RowBox[{"\[Eta]3", "->", "\[Eta]S1"}], ",", 
       RowBox[{"\[Eta]4", "->", "\[Eta]S1"}], ",", 
       RowBox[{"p0", "->", 
        RowBox[{"ProbFn", "[", "0", "]"}]}], ",", 
       RowBox[{"p1", "->", 
        RowBox[{"ProbFn", "[", "1", "]"}]}], ",", 
       RowBox[{"p2", "->", 
        RowBox[{"ProbFn", "[", "2", "]"}]}]}], "}"}]}], "]"}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"SPDCSrc2", "=", 
   RowBox[{"Simplify", "[", 
    RowBox[{"SPDC4", "/.", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"\[Eta]1", "->", "\[Eta]S2"}], ",", 
       RowBox[{"\[Eta]2", "->", "\[Eta]S2"}], ",", 
       RowBox[{"\[Eta]3", "->", "\[Eta]B"}], ",", 
       RowBox[{"\[Eta]4", "->", "\[Eta]B"}], ",", 
       RowBox[{"p0", "->", 
        RowBox[{"ProbFn", "[", "0", "]"}]}], ",", 
       RowBox[{"p1", "->", 
        RowBox[{"ProbFn", "[", "1", "]"}]}], ",", 
       RowBox[{"p2", "->", 
        RowBox[{"ProbFn", "[", "2", "]"}]}]}], "}"}]}], "]"}]}], 
  ";"}]}], "Input",
 CellChangeTimes->{{3.918043619323297*^9, 3.9180436778813148`*^9}, {
  3.918044167506589*^9, 3.918044190282659*^9}},
 CellLabel->"In[95]:=",ExpressionUUID->"494ddb28-4101-45a9-8b30-ac2caa82a41d"],

Cell[BoxData[
 RowBox[{
  RowBox[{"Vec2SPDC", "=", 
   RowBox[{"ArrayReshape", "[", 
    RowBox[{
     RowBox[{"Table", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
        "k1", ",", "k2", ",", "k3", ",", "k4", ",", "k5", ",", "k6", ",", 
         "k7", ",", "k8"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"k1", ",", "0", ",", "2"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"k2", ",", "0", ",", "2"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"k3", ",", "0", ",", "2"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"k4", ",", "0", ",", "2"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"k5", ",", "0", ",", "2"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"k6", ",", "0", ",", "2"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"k7", ",", "0", ",", "2"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"k8", ",", "0", ",", "2"}], "}"}]}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"6561", ",", "8"}], "}"}]}], "]"}]}], ";"}]], "Input",
 CellChangeTimes->{{3.918043690621052*^9, 3.918043740861807*^9}},
 CellLabel->"In[97]:=",ExpressionUUID->"94bafb7c-3e87-45ba-9b78-dba974ee9fe2"],

Cell[BoxData[
 RowBox[{
  RowBox[{"SPDC2Src", "=", 
   RowBox[{"Kron", "[", 
    RowBox[{"SPDCSrc1", ",", "SPDCSrc2"}], "]"}]}], ";"}]], "Input",
 CellChangeTimes->{{3.91804375041545*^9, 3.918043774591054*^9}, 
   3.918043808475403*^9},
 CellLabel->"In[98]:=",ExpressionUUID->"66a5b282-e8c5-4e95-b9f2-0ab48ba1ffd6"],

Cell["Applying the beamsplitter between mode 3 and 5", "Text",
 CellChangeTimes->{{3.918043241036032*^9, 3.918043268917938*^9}, {
  3.918056287837054*^9, 3.918056300796493*^9}, {3.924212870610861*^9, 
  3.924212873903042*^9}},
 Background->RGBColor[
  0.87, 0.94, 1],ExpressionUUID->"715da64c-4ad6-4836-9f65-18ed765621f5"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"SPDC2Step1", "=", 
   RowBox[{"DMatMMBSAggregateSp", "[", 
    RowBox[{"SPDC2Src", ",", "Vec2SPDC", ",", "3", ",", "5", ",", 
     RowBox[{"\[Pi]", "/", "4"}], ",", "0"}], "]"}]}], ";"}]], "Input",
 CellChangeTimes->{{3.918043855196621*^9, 3.918043867248392*^9}, {
  3.918043947806035*^9, 3.918043973423581*^9}, {3.9180441410852547`*^9, 
  3.918044141366701*^9}},
 CellLabel->"In[99]:=",ExpressionUUID->"9b3c04fe-b48d-46af-b892-04c9331c5a5e"],

Cell[BoxData["6400"], "Print",
 CellChangeTimes->{3.91804414300982*^9, 3.918044198719634*^9, 
  3.92426220206511*^9},
 CellLabel->
  "During evaluation of \
In[99]:=",ExpressionUUID->"6b84037e-7aba-4445-b134-3b9d7f7493e6"]
}, Open  ]],

Cell["Applying the beamsplitter between mode 4 and 6", "Text",
 CellChangeTimes->{{3.918043241036032*^9, 3.918043268917938*^9}, {
   3.918056287837054*^9, 3.9180563118070803`*^9}, 3.924212880680202*^9},
 Background->RGBColor[
  0.87, 0.94, 1],ExpressionUUID->"59a13f53-159d-46ae-8c58-96dbb5603c6a"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"SPDC2Step2", "=", 
   RowBox[{"DMatMMBSAggregateSp", "[", 
    RowBox[{"SPDC2Step1", ",", "Vec2SPDC", ",", "4", ",", "6", ",", 
     RowBox[{"\[Pi]", "/", "4"}], ",", "0"}], "]"}]}], ";"}]], "Input",
 CellChangeTimes->{{3.918047864599244*^9, 3.918047871990369*^9}},
 CellLabel->
  "In[100]:=",ExpressionUUID->"2f071340-8e4e-4e0a-83f5-77c6d5ee035c"],

Cell[BoxData["19936"], "Print",
 CellChangeTimes->{3.918047875431568*^9, 3.924264739612553*^9},
 CellLabel->
  "During evaluation of \
In[100]:=",ExpressionUUID->"c6036198-48cd-4854-a725-8f7d2c9f347d"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Postselected State", "Subsection",
 CellChangeTimes->{{3.918054605631219*^9, 
  3.918054612446316*^9}},ExpressionUUID->"6ab55a5d-18c5-4e93-ace3-\
829f8426d2ee"],

Cell[BoxData[
 RowBox[{
  RowBox[{"FilterDensMat", "[", 
   RowBox[{"M_", ",", "filtLoc_"}], "]"}], ":=", 
  RowBox[{"Table", "[", 
   RowBox[{
    RowBox[{"M", "[", 
     RowBox[{"[", 
      RowBox[{
       RowBox[{"filtLoc", "[", 
        RowBox[{"[", "i", "]"}], "]"}], ",", 
       RowBox[{"filtLoc", "[", 
        RowBox[{"[", "j", "]"}], "]"}]}], "]"}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"i", ",", "1", ",", 
      RowBox[{"Length", "[", "filtLoc", "]"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"j", ",", "1", ",", 
      RowBox[{"Length", "[", "filtLoc", "]"}]}], "}"}]}], "]"}]}]], "Code",
 CellChangeTimes->{{3.832089330531717*^9, 3.832089444671986*^9}, {
  3.892391142904565*^9, 3.892391145559162*^9}},
 CellLabel->
  "In[101]:=",ExpressionUUID->"8142d618-fa20-4b85-b810-506e1b674602"],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"PostselectPattern", "[", 
    RowBox[{"pattern__", ",", "pos__", ",", "DMat_", ",", "lvec__"}], "]"}], ":=", 
   RowBox[{"Module", "[", 
    RowBox[{
     RowBox[{"{", "patternList", "}"}], ",", "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{"patternList", "=", 
       RowBox[{"{", "}"}]}], ";", "\[IndentingNewLine]", 
      RowBox[{"Do", "[", 
       RowBox[{
        RowBox[{"If", "[", 
         RowBox[{
          RowBox[{
           RowBox[{"lvec", "[", 
            RowBox[{"[", 
             RowBox[{"i", ",", "pos"}], "]"}], "]"}], "===", "pattern"}], ",", 
          RowBox[{
           RowBox[{"AppendTo", "[", 
            RowBox[{"patternList", ",", "i"}], "]"}], ";"}]}], "]"}], ",", 
        RowBox[{"{", 
         RowBox[{"i", ",", "1", ",", 
          RowBox[{"Length", "[", "lvec", "]"}]}], "}"}]}], "]"}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"Return", "[", 
       RowBox[{"FilterDensMat", "[", 
        RowBox[{"DMat", ",", "patternList"}], "]"}], "]"}]}]}], 
    "\[IndentingNewLine]", "]"}]}], "\[IndentingNewLine]"}]], "Input",
 CellChangeTimes->{{3.918054702321494*^9, 3.91805487693556*^9}},
 CellLabel->
  "In[102]:=",ExpressionUUID->"c3c53afe-b7f6-4e06-bb21-56df2d291853"],

Cell["\<\
Filtering out the state for no excess noise in the detectors (P_d = 0)\
\>", "Text",
 CellChangeTimes->{{3.9180555143471527`*^9, 3.9180555509793797`*^9}},
 Background->RGBColor[
  0.87, 0.94, 1],ExpressionUUID->"dfd4cb66-4b6c-41a9-af6b-8dbbc2992d62"],

Cell[BoxData[
 RowBox[{
  RowBox[{"HMCSDMatNoNoise", "=", 
   RowBox[{"PostselectPattern", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "1", ",", "0", ",", "0"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"3", ",", "4", ",", "5", ",", "6"}], "}"}], ",", "SPDC2Step2", 
     ",", "Vec2SPDC"}], "]"}]}], ";"}]], "Input",
 CellChangeTimes->{{3.918054878965897*^9, 3.918054932819923*^9}, {
  3.918055033963292*^9, 3.918055121669575*^9}, {3.91805515299825*^9, 
  3.918055201031261*^9}},
 CellLabel->
  "In[103]:=",ExpressionUUID->"39b24af6-f495-4a35-8e63-95e9819e44f5"],

Cell["\<\
Filtering out the state for excess noise in the detectors (P_d > 0)\
\>", "Text",
 CellChangeTimes->{{3.9180555143471527`*^9, 3.918055549156489*^9}},
 Background->RGBColor[
  0.87, 0.94, 1],ExpressionUUID->"ac45fa38-0ab9-44d8-b4af-98ce671b239a"],

Cell[BoxData[
 RowBox[{
  RowBox[{"HMCSDMatNoisy", "=", 
   RowBox[{
    RowBox[{
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{"1", "-", "Pd"}], ")"}], "4"], 
     RowBox[{"PostselectPattern", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"1", ",", "1", ",", "0", ",", "0"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"3", ",", "4", ",", "5", ",", "6"}], "}"}], ",", "SPDC2Step2",
        ",", "Vec2SPDC"}], "]"}]}], "+", 
    RowBox[{
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{"1", "-", "Pd"}], ")"}], "3"], "Pd", " ", 
     RowBox[{"PostselectPattern", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"1", ",", "0", ",", "0", ",", "0"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"3", ",", "4", ",", "5", ",", "6"}], "}"}], ",", "SPDC2Step2",
        ",", "Vec2SPDC"}], "]"}]}], "+", 
    RowBox[{
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{"1", "-", "Pd"}], ")"}], "3"], "Pd", " ", 
     RowBox[{"PostselectPattern", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"0", ",", "1", ",", "0", ",", "0"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"3", ",", "4", ",", "5", ",", "6"}], "}"}], ",", "SPDC2Step2",
        ",", "Vec2SPDC"}], "]"}]}], "+", 
    RowBox[{
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{"1", "-", "Pd"}], ")"}], "2"], 
     SuperscriptBox["Pd", "2"], " ", 
     RowBox[{"PostselectPattern", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"0", ",", "0", ",", "0", ",", "0"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"3", ",", "4", ",", "5", ",", "6"}], "}"}], ",", "SPDC2Step2",
        ",", "Vec2SPDC"}], "]"}]}]}]}], ";"}]], "Input",
 CellChangeTimes->{{3.918055202643527*^9, 3.918055268829977*^9}},
 CellLabel->
  "In[104]:=",ExpressionUUID->"db4965c6-acae-4dd1-b124-0af8977843c7"],

Cell["\<\
Array of vectors for the heralded multiplexed cascaded source state\
\>", "Text",
 CellChangeTimes->{{3.924212999025107*^9, 3.924213042100634*^9}},
 Background->RGBColor[
  0.87, 0.94, 1],ExpressionUUID->"fcf9e3fe-37e9-418a-93a6-c111cdfc6201"],

Cell[BoxData[
 RowBox[{
  RowBox[{"VecHMCS", "=", 
   RowBox[{"ArrayReshape", "[", 
    RowBox[{
     RowBox[{"Table", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"k1", ",", "k2", ",", "k3", ",", "k4"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"k1", ",", "0", ",", "2"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"k2", ",", "0", ",", "2"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"k3", ",", "0", ",", "2"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"k4", ",", "0", ",", "2"}], "}"}]}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"81", ",", "4"}], "}"}]}], "]"}]}], ";"}]], "Input",
 CellChangeTimes->{{3.918055423118355*^9, 3.918055442042206*^9}},
 CellLabel->
  "In[105]:=",ExpressionUUID->"2e10a4af-f38a-4d70-9c45-84c24ebd6e70"],

Cell["\<\
Locating the positions of the desirable Bell pair like terms (positions 13, \
29)\
\>", "Text",
 CellChangeTimes->{{3.918055457600894*^9, 3.918055476869845*^9}, {
  3.9242129523396883`*^9, 3.924212959756578*^9}},
 Background->RGBColor[
  0.87, 0.94, 1],ExpressionUUID->"b1c43e5e-9bec-4428-b9fc-1efcbdfc9243"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Position", "[", 
  RowBox[{"VecHMCS", ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "0", ",", "0", ",", "1"}], "}"}]}], "]"}]], "Input",
 CellChangeTimes->{{3.918055478852762*^9, 3.91805548733659*^9}},
 CellLabel->
  "In[106]:=",ExpressionUUID->"d836e27a-9075-41c8-9dec-eb0423807b16"],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", "29", "}"}], "}"}]], "Output",
 CellChangeTimes->{3.918055488098604*^9, 3.924279227586866*^9},
 CellLabel->
  "Out[106]=",ExpressionUUID->"ba590d77-6bfa-4c36-8458-f5dc815073c1"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Position", "[", 
  RowBox[{"VecHMCS", ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "1", ",", "1", ",", "0"}], "}"}]}], "]"}]], "Input",
 CellChangeTimes->{{3.918055501546056*^9, 3.918055504504363*^9}},
 CellLabel->
  "In[107]:=",ExpressionUUID->"a128d2f9-6f7d-4757-b11f-4a513cfbae7c"],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", "13", "}"}], "}"}]], "Output",
 CellChangeTimes->{3.918055505435509*^9, 3.92427922759346*^9},
 CellLabel->
  "Out[107]=",ExpressionUUID->"dfd62353-130f-4ec2-9fc9-1eba2011b888"]
}, Open  ]],

Cell["\<\
Locating the positions of the undesirable terms (positions 5, 37)\
\>", "Text",
 CellChangeTimes->{{3.918055457600894*^9, 3.918055476869845*^9}, {
  3.918055584571102*^9, 3.918055596038109*^9}, {3.9242129634619007`*^9, 
  3.924212970093375*^9}},
 Background->RGBColor[
  0.87, 0.94, 1],ExpressionUUID->"284e6285-bf17-4231-91df-1428f879ecc6"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Position", "[", 
  RowBox[{"VecHMCS", ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "1", ",", "0", ",", "0"}], "}"}]}], "]"}]], "Input",
 CellChangeTimes->{{3.918055478852762*^9, 3.91805548733659*^9}, {
  3.918055606816451*^9, 3.918055608684692*^9}},
 CellLabel->
  "In[108]:=",ExpressionUUID->"430be553-3653-4ba7-ad53-d41fc1ddbbf4"],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", "37", "}"}], "}"}]], "Output",
 CellChangeTimes->{3.918055488098604*^9, 3.918055614983116*^9, 
  3.9242792276234617`*^9},
 CellLabel->
  "Out[108]=",ExpressionUUID->"e13ed3eb-8fae-4b5f-a827-62d9c3ad9aa4"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Position", "[", 
  RowBox[{"VecHMCS", ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "1", ",", "1"}], "}"}]}], "]"}]], "Input",
 CellChangeTimes->{{3.918055501546056*^9, 3.918055504504363*^9}, {
  3.918055610642215*^9, 3.918055613296774*^9}},
 CellLabel->
  "In[109]:=",ExpressionUUID->"fdfdae3e-5079-43e9-a1de-7d4fc37366c3"],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", "5", "}"}], "}"}]], "Output",
 CellChangeTimes->{3.918055505435509*^9, 3.918055616711834*^9, 
  3.924279227629816*^9},
 CellLabel->
  "Out[109]=",ExpressionUUID->"2c4dbd29-25c2-4d6f-b8e9-3d2ffa97d845"]
}, Open  ]],

Cell["Including swap visibility", "Text",
 CellChangeTimes->{{3.92421556890211*^9, 3.924215575182665*^9}},
 Background->RGBColor[
  0.87, 0.94, 1],ExpressionUUID->"5478263b-3def-4c6f-8279-77dd211dd64d"],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"HMCSDMatNoisy", "[", 
    RowBox[{"[", 
     RowBox[{"13", ",", "29"}], "]"}], "]"}], "=", 
   RowBox[{
    RowBox[{"HMCSDMatNoisy", "[", 
     RowBox[{"[", 
      RowBox[{"13", ",", "29"}], "]"}], "]"}], "*", "vis"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"HMCSDMatNoisy", "[", 
    RowBox[{"[", 
     RowBox[{"29", ",", "13"}], "]"}], "]"}], "=", 
   RowBox[{
    RowBox[{"HMCSDMatNoisy", "[", 
     RowBox[{"[", 
      RowBox[{"29", ",", "13"}], "]"}], "]"}], "*", "vis"}]}], 
  ";"}]}], "Input",
 CellChangeTimes->{{3.9242155270838127`*^9, 3.92421560916635*^9}, {
  3.924262222597888*^9, 3.924262227787078*^9}, {3.92427923089928*^9, 
  3.9242792315680637`*^9}},ExpressionUUID->"bd670a4d-8793-4280-a3da-\
4a4627e150cd"],

Cell[CellGroupData[{

Cell["Saving File", "Subsubsection",
 CellChangeTimes->{{3.924212912304859*^9, 
  3.924212916356223*^9}},ExpressionUUID->"9a52463c-4221-49b9-b2e7-\
d166328fe523"],

Cell[BoxData[
 RowBox[{
  RowBox[{"HMCSAfterLossMATLAB", "=", 
   RowBox[{"HMCSDMatNoisy", "/.", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"\[Eta]A", "->", "eA"}], ",", 
      RowBox[{"\[Eta]B", "->", "eB"}], ",", 
      RowBox[{"\[Eta]S1", "->", "eC1"}], ",", 
      RowBox[{"\[Eta]S2", "->", "eC2"}]}], "}"}]}]}], ";"}]], "Input",
 CellChangeTimes->{{3.918064055655372*^9, 3.9180640707345657`*^9}, {
   3.91808373030007*^9, 3.91808373323862*^9}, {3.924212943312967*^9, 
   3.924212945495901*^9}, 
   3.949874740314866*^9},ExpressionUUID->"cb455290-3bd5-406e-92b3-\
0f0ce0e8de27"],

Cell[BoxData[
 RowBox[{"WriteMatlab", "[", 
  RowBox[{"HMCSAfterLossMATLAB", ",", " ", "\"\<dens_mat_HMCS.m\>\""}], 
  "]"}]], "Input",
 CellChangeTimes->{{3.924212779548608*^9, 3.924212854528118*^9}, {
  3.92421292730914*^9, 
  3.924212935330715*^9}},ExpressionUUID->"b3f0b43a-2a5e-4dd4-ad1f-\
945bdf53c703"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Testing", "Subsubsection",
 CellChangeTimes->{{3.924261755712817*^9, 3.9242617575039387`*^9}, 
   3.9242621276538*^9},ExpressionUUID->"4654884e-5c04-4e9d-86f7-8d5c099dcb3e"],

Cell["Test Case 1 : State with no losses and Ns  = 1e-3; vis=1", "Text",
 CellChangeTimes->{{3.9242617623380136`*^9, 3.924261780297464*^9}, 
   3.9242792750482264`*^9, {3.924280067226576*^9, 3.9242800686116333`*^9}},
 Background->RGBColor[
  0.87, 0.94, 1],ExpressionUUID->"0ab49d45-52d1-466e-bed5-fbfce27ee2f9"],

Cell[BoxData[{
 RowBox[{
  RowBox[{"HMCSTest1", "=", 
   RowBox[{"HMCSAfterLossMATLAB", "/.", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"eA", "->", "1"}], ",", 
      RowBox[{"eB", "->", "1."}], ",", 
      RowBox[{"eC1", "->", "1"}], ",", 
      RowBox[{"eC2", "->", "1"}], ",", 
      RowBox[{"Ns", "->", 
       SuperscriptBox["10", 
        RowBox[{"-", "3"}]]}], ",", 
      RowBox[{"vis", "->", "1"}]}], "}"}]}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"Export", "[", 
   RowBox[{"\"\<HMCSTest1.txt\>\"", ",", "HMCSTest1", ",", "\"\<CSV\>\""}], 
   "]"}], ";"}]}], "Input",
 CellChangeTimes->{{3.924261784015253*^9, 3.924261865368359*^9}, {
   3.92426191657752*^9, 3.9242619257186823`*^9}, 3.924261973524811*^9, {
   3.924279289960496*^9, 3.924279312597992*^9}, 
   3.92428007426291*^9},ExpressionUUID->"5807a433-3c64-4219-9132-\
639b019da95a"],

Cell["\<\
Test Case 2 : State with no coupling losses, external arms loss of 10 dB and \
Ns  = 1e-3; vis =1\
\>", "Text",
 CellChangeTimes->{{3.9242617623380136`*^9, 3.924261780297464*^9}, 
   3.9242792750482264`*^9, {3.924279340137474*^9, 3.92427936194765*^9}, {
   3.924280064286212*^9, 3.924280065417372*^9}},
 Background->RGBColor[
  0.87, 0.94, 1],ExpressionUUID->"eec3dc5e-b370-4e26-be83-214010f74f8a"],

Cell[BoxData[{
 RowBox[{
  RowBox[{"HMCSTest2", "=", 
   RowBox[{"HMCSAfterLossMATLAB", "/.", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"eA", "->", "0.1"}], ",", 
      RowBox[{"eB", "->", "0.1"}], ",", 
      RowBox[{"eC1", "->", "1"}], ",", 
      RowBox[{"eC2", "->", "1"}], ",", 
      RowBox[{"Ns", "->", 
       SuperscriptBox["10", 
        RowBox[{"-", "3"}]]}], ",", 
      RowBox[{"vis", "->", "1"}]}], "}"}]}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"Export", "[", 
   RowBox[{"\"\<HMCSTest2.txt\>\"", ",", "HMCSTest2", ",", "\"\<CSV\>\""}], 
   "]"}], ";"}]}], "Input",
 CellChangeTimes->{{3.924261784015253*^9, 3.924261865368359*^9}, {
   3.92426191657752*^9, 3.9242619257186823`*^9}, 3.924261973524811*^9, {
   3.924279289960496*^9, 3.924279312597992*^9}, {3.924279365455097*^9, 
   3.924279375473784*^9}, 
   3.924280073006481*^9},ExpressionUUID->"a9486e34-5135-48ea-8b64-\
7cc1ebab9b2b"],

Cell["\<\
Test Case 1 : State with 0.9 coupling efficiency, external arms loss of 10 dB \
and Ns  = 1e-3; vis=1\
\>", "Text",
 CellChangeTimes->{{3.9242617623380136`*^9, 3.924261780297464*^9}, 
   3.9242792750482264`*^9, {3.92427938822538*^9, 3.924279395622239*^9}, {
   3.924280061114485*^9, 3.924280062243267*^9}},
 Background->RGBColor[
  0.87, 0.94, 1],ExpressionUUID->"3f97d01e-213f-4e7d-8ed3-69387f3547f8"],

Cell[BoxData[{
 RowBox[{
  RowBox[{"HMCSTest3", "=", 
   RowBox[{"HMCSAfterLossMATLAB", "/.", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"eA", "->", "0.1"}], ",", 
      RowBox[{"eB", "->", "0.1"}], ",", 
      RowBox[{"eC1", "->", "0.9"}], ",", 
      RowBox[{"eC2", "->", "0.9"}], ",", 
      RowBox[{"Ns", "->", 
       SuperscriptBox["10", 
        RowBox[{"-", "3"}]]}], ",", 
      RowBox[{"vis", "->", "1"}]}], "}"}]}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"Export", "[", 
   RowBox[{"\"\<HMCSTest3.txt\>\"", ",", "HMCSTest3", ",", "\"\<CSV\>\""}], 
   "]"}], ";"}]}], "Input",
 CellChangeTimes->{{3.924261784015253*^9, 3.924261865368359*^9}, {
   3.92426191657752*^9, 3.9242619257186823`*^9}, 3.924261973524811*^9, {
   3.924279289960496*^9, 3.924279312597992*^9}, {3.9242794010490828`*^9, 
   3.924279420737558*^9}, {3.9242800499641647`*^9, 
   3.924280052042327*^9}},ExpressionUUID->"113760a1-21d4-45e4-bb62-\
992b3938be81"],

Cell["\<\
Test Case 2 : State with no coupling losses, external arms loss of 20 dB and \
Ns  = 1e-1; vis=0.95\
\>", "Text",
 CellChangeTimes->{{3.9242617623380136`*^9, 3.924261780297464*^9}, 
   3.9242792750482264`*^9, {3.924279340137474*^9, 3.92427936194765*^9}, {
   3.924279430090908*^9, 3.924279433504167*^9}, {3.92428005416015*^9, 
   3.924280058817708*^9}},
 Background->RGBColor[
  0.87, 0.94, 1],ExpressionUUID->"040b9f80-a710-4f0c-80d0-cff0ab3f17e0"],

Cell[BoxData[{
 RowBox[{
  RowBox[{"HMCSTest4", "=", 
   RowBox[{"HMCSAfterLossMATLAB", "/.", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"eA", "->", "0.01"}], ",", 
      RowBox[{"eB", "->", "0.01"}], ",", 
      RowBox[{"eC1", "->", "1"}], ",", 
      RowBox[{"eC2", "->", "1"}], ",", 
      RowBox[{"Ns", "->", 
       SuperscriptBox["10", 
        RowBox[{"-", "1"}]]}], ",", 
      RowBox[{"vis", "->", "0.95"}]}], "}"}]}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"Export", "[", 
   RowBox[{"\"\<HMCSTest4.txt\>\"", ",", "HMCSTest4", ",", "\"\<CSV\>\""}], 
   "]"}], ";"}]}], "Input",
 CellChangeTimes->{{3.924261784015253*^9, 3.924261865368359*^9}, {
   3.92426191657752*^9, 3.9242619257186823`*^9}, 3.924261973524811*^9, {
   3.924279289960496*^9, 3.924279312597992*^9}, {3.924279365455097*^9, 
   3.924279375473784*^9}, {3.924279422743237*^9, 3.924279441635962*^9}, {
   3.924280081176887*^9, 
   3.924280086396431*^9}},ExpressionUUID->"712a3c84-5f88-4a37-8f68-\
af4454118e48"]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}, Open  ]]
},
WindowSize->{Full, Full},
WindowMargins->{{556, Automatic}, {Automatic, 50}},
FrontEndVersion->"14.0 for Linux x86 (64-bit) (December 12, 2023)",
StyleDefinitions->"Default.nb",
ExpressionUUID->"2dee8d45-720e-461a-8d2e-f606336d5fb5"
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{}
*)
(*CellTagsIndex
CellTagsIndex->{}
*)
(*NotebookFileOutline
Notebook[{
Cell[CellGroupData[{
Cell[580, 22, 363, 8, 160, "Title",ExpressionUUID->"cb50704f-3a2b-45c1-8acd-20138b4efdd7"],
Cell[946, 32, 1972, 35, 327, "Text",ExpressionUUID->"4de4db4e-57b9-4d39-bc4e-87c589bd4855"],
Cell[2921, 69, 199, 3, 29, "Input",ExpressionUUID->"bf235b78-c871-49a6-a5e2-52a4eebcbd1a"],
Cell[CellGroupData[{
Cell[3145, 76, 154, 3, 69, "Chapter",ExpressionUUID->"14d8adce-587d-402b-8b94-98c892f17fef"],
Cell[CellGroupData[{
Cell[3324, 83, 273, 4, 67, "Section",ExpressionUUID->"f17212c3-ace8-49f4-8193-88aafb766cd0"],
Cell[3600, 89, 222, 3, 51, "Text",ExpressionUUID->"db6f0739-c15f-48af-ad90-b89575eb79e5"],
Cell[3825, 94, 535, 15, 52, "Input",ExpressionUUID->"48971438-373a-4c62-8eee-2be550f74bab"],
Cell[4363, 111, 564, 16, 52, "Input",ExpressionUUID->"01868f09-e100-4cdd-af78-d6e3afaa06c9"],
Cell[4930, 129, 299, 6, 30, "Input",ExpressionUUID->"bb462150-f922-439c-9b80-e98e0e0b0702"],
Cell[5232, 137, 702, 18, 30, "Input",ExpressionUUID->"1eaecd85-5f92-4a17-9ca4-f9330716ede2"],
Cell[5937, 157, 431, 8, 51, "Text",ExpressionUUID->"33d9b657-2781-44f6-8b62-6adbcbe34368"],
Cell[6371, 167, 525, 13, 30, "Input",ExpressionUUID->"5bae2af4-eeac-4ecf-bc3a-3cb4a6fa056f"],
Cell[6899, 182, 238, 3, 51, "Text",ExpressionUUID->"161efc38-466b-410e-92a1-857f16f08bcb"],
Cell[7140, 187, 777, 17, 30, "Input",ExpressionUUID->"8eb3095f-1c4a-4b98-adb7-8a6c09ff8978"],
Cell[7920, 206, 733, 17, 30, "Input",ExpressionUUID->"aa437d8a-5930-46ad-89c3-10c60cc6de73"],
Cell[8656, 225, 733, 17, 30, "Input",ExpressionUUID->"038a0f56-4648-4bc9-aebf-05d21e7974f0"],
Cell[9392, 244, 397, 8, 30, "Input",ExpressionUUID->"92f854ae-848c-47e3-8374-0a34448a794d"],
Cell[9792, 254, 352, 7, 30, "Input",ExpressionUUID->"7c464376-8264-4056-bdf0-3f4ec6aff398"],
Cell[10147, 263, 185, 3, 35, "Text",ExpressionUUID->"141b6aa7-187a-4bb9-9e99-9277c5d3bcba"],
Cell[10335, 268, 383, 9, 46, "Input",ExpressionUUID->"bd5fec39-ef74-47ff-acfb-8c8dfec8cb98"],
Cell[10721, 279, 1545, 42, 152, "Input",ExpressionUUID->"095cd585-8755-4647-b5c7-e3c2a8eaf759"]
}, Closed]],
Cell[CellGroupData[{
Cell[12303, 326, 312, 5, 53, "Section",ExpressionUUID->"31d7f278-dfda-4c19-8d84-52aaf36b8f39"],
Cell[CellGroupData[{
Cell[12640, 335, 214, 4, 54, "Subsection",ExpressionUUID->"91b59b93-b46e-486d-9c20-4663a55d7a4f"],
Cell[12857, 341, 329, 5, 51, "Text",ExpressionUUID->"a5f1009b-b10c-4490-b854-515b27446531"],
Cell[13189, 348, 2582, 70, 140, "Input",ExpressionUUID->"7ea707f7-ba85-4bcd-8c85-16732af22afe"],
Cell[15774, 420, 1214, 22, 235, "Text",ExpressionUUID->"b1fd8761-3f65-433e-9de5-a09625902f38"],
Cell[16991, 444, 4672, 115, 262, "Input",ExpressionUUID->"b41baffb-ed76-4b5a-a292-52a355a458ce"],
Cell[21666, 561, 1055, 19, 212, "Text",ExpressionUUID->"7e31a995-e53a-4eab-99e4-35f707513500"],
Cell[22724, 582, 2707, 64, 220, "Input",ExpressionUUID->"da95d9ea-987b-4fd6-bedb-470a8bc7051d"],
Cell[25434, 648, 269, 6, 51, "Text",ExpressionUUID->"8f0f4971-1a22-45c1-a753-e828597bd194"],
Cell[25706, 656, 5587, 131, 325, "Input",ExpressionUUID->"48b7b6ca-717e-4fa5-b325-980225b0714b"],
Cell[31296, 789, 4148, 88, 283, "Input",ExpressionUUID->"e2880865-33ff-4665-a1a3-b41a09672ebe"]
}, Open  ]],
Cell[CellGroupData[{
Cell[35481, 882, 161, 3, 54, "Subsection",ExpressionUUID->"836223b7-7bd2-456a-8ea0-760432d91a63"],
Cell[35645, 887, 431, 11, 143, "Text",ExpressionUUID->"500c17ab-d9db-431b-9c3b-77d8c13d82a1"],
Cell[36079, 900, 958, 27, 60, "Input",ExpressionUUID->"2f30392d-792e-4864-b1c9-e1d619e45772"],
Cell[37040, 929, 653, 15, 189, "Text",ExpressionUUID->"74ff8bcc-e080-4828-a529-211a15892aef"],
Cell[37696, 946, 1522, 38, 136, "Input",ExpressionUUID->"bda1dfbb-dcff-41c1-9af6-d71407e2b2f0"],
Cell[39221, 986, 600, 14, 212, "Text",ExpressionUUID->"449106d1-de1e-4cdb-86f7-c873c923fe3a"],
Cell[39824, 1002, 4314, 106, 346, "Input",ExpressionUUID->"595ac4e2-67d2-4596-adbe-ad4dbcd4210c"]
}, Open  ]],
Cell[CellGroupData[{
Cell[44175, 1113, 155, 3, 54, "Subsection",ExpressionUUID->"f74dba2f-1e96-4cd6-be98-981969e69e01"],
Cell[44333, 1118, 202, 3, 51, "Text",ExpressionUUID->"43168903-8a53-49a3-8d52-e3b4b9412f3a"],
Cell[44538, 1123, 680, 20, 52, "Input",ExpressionUUID->"39183672-4e9a-4d10-b919-6ef86aafc324"],
Cell[CellGroupData[{
Cell[45243, 1147, 353, 7, 30, "Input",ExpressionUUID->"c19aab7b-bf06-4236-bff5-bd7496fc25e9"],
Cell[45599, 1156, 996, 29, 70, "Output",ExpressionUUID->"e41ebb7c-416e-4263-bcfe-630a2d21140c"]
}, Open  ]],
Cell[CellGroupData[{
Cell[46632, 1190, 666, 18, 52, "Input",ExpressionUUID->"a6153760-081e-403b-8e86-459bf34c34df"],
Cell[47301, 1210, 728, 22, 70, "Output",ExpressionUUID->"19c5d0b5-c956-4fd5-ad53-4e3dc505bf69"]
}, Open  ]],
Cell[CellGroupData[{
Cell[48066, 1237, 778, 22, 52, "Input",ExpressionUUID->"fd4f4e7c-7bb0-407c-b6de-2c392a3ee755"],
Cell[48847, 1261, 1368, 44, 70, "Output",ExpressionUUID->"6a7b5791-e2b8-4453-a287-28d15174de17"]
}, Open  ]],
Cell[50230, 1308, 249, 4, 51, "Text",ExpressionUUID->"431108c5-df9d-494f-a896-1546b2503780"],
Cell[50482, 1314, 1055, 28, 52, "Input",ExpressionUUID->"ab2554b5-fc04-4bee-b42a-651551e2bf83"],
Cell[CellGroupData[{
Cell[51562, 1346, 281, 6, 30, "Input",ExpressionUUID->"39368815-9a78-4dff-b0de-07b8745411f9"],
Cell[51846, 1354, 1369, 37, 70, "Output",ExpressionUUID->"1fb2b3bb-1061-47db-b0bb-a91ff793d766"]
}, Open  ]],
Cell[CellGroupData[{
Cell[53252, 1396, 882, 22, 52, "Input",ExpressionUUID->"8d2502b4-505b-43e4-9376-8c9351f889e7"],
Cell[54137, 1420, 226, 3, 70, "Output",ExpressionUUID->"3a5addd4-cc73-480f-98f5-2a5a4274c51a"]
}, Open  ]]
}, Closed]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[54436, 1431, 195, 3, 69, "Chapter",ExpressionUUID->"78b09564-4fdc-40bc-b187-55abd708f9a1"],
Cell[CellGroupData[{
Cell[54656, 1438, 174, 3, 67, "Section",ExpressionUUID->"ea26b07c-321f-40a9-b8e7-dd87cb354b10"],
Cell[54833, 1443, 212, 3, 51, "Text",ExpressionUUID->"67c2e5ae-5d0f-4b86-a10d-1487efa5bf54"],
Cell[55048, 1448, 458, 13, 48, "Input",ExpressionUUID->"972dba28-8c79-4229-a144-eb7e4c772a79"],
Cell[55509, 1463, 300, 5, 51, "Text",ExpressionUUID->"211c00c8-c138-4e78-9cd5-ce342c54ce76"],
Cell[55812, 1470, 1799, 51, 84, "Input",ExpressionUUID->"15209ca1-c1dd-46ab-a25f-ab145a7bcf23"],
Cell[57614, 1523, 359, 5, 51, "Text",ExpressionUUID->"2d92e80e-56f5-44ea-89a2-c0e3e48c5d8b"],
Cell[57976, 1530, 288, 6, 29, "Input",ExpressionUUID->"401c8d67-790c-4c97-bfd9-bc95372f9578"],
Cell[58267, 1538, 313, 7, 51, "Text",ExpressionUUID->"d1e1f06d-48ff-475f-85a2-feddcc1a2ae3"],
Cell[58583, 1547, 781, 20, 29, "Input",ExpressionUUID->"ac1da23b-0ae7-4cd0-b51c-ee30a6a9600d"],
Cell[59367, 1569, 966, 25, 110, "Input",ExpressionUUID->"ab886123-0acf-4364-9241-036550aa3b2b"],
Cell[60336, 1596, 256, 5, 51, "Text",ExpressionUUID->"cd3af855-7d76-4d63-a308-31e64cd4f85b"],
Cell[60595, 1603, 770, 18, 29, "Input",ExpressionUUID->"2d71162a-8ba7-4ee9-b734-8d03685e1bb2"],
Cell[CellGroupData[{
Cell[61390, 1625, 162, 3, 45, "Subsubsection",ExpressionUUID->"b5fb754e-77ac-436f-a6c4-49c3ec9207ac"],
Cell[61555, 1630, 701, 17, 29, "Input",ExpressionUUID->"63a8778b-b776-40ab-a55e-f257326fbb31"],
Cell[62259, 1649, 280, 5, 31, "Input",ExpressionUUID->"654c5a02-2701-47cd-a8b8-3f0bb6ac6425"]
}, Open  ]],
Cell[CellGroupData[{
Cell[62576, 1659, 179, 2, 45, "Subsubsection",ExpressionUUID->"f1f53dfc-10cd-4693-b7dc-66271edad814"],
Cell[62758, 1663, 230, 3, 51, "Text",ExpressionUUID->"5a7a8759-980f-494b-87e6-36c3a3e1f7b0"],
Cell[62991, 1668, 695, 17, 52, "Input",ExpressionUUID->"93c407ac-39de-4288-9f7f-2c428630cb22"],
Cell[63689, 1687, 310, 6, 51, "Text",ExpressionUUID->"be94e640-f9a7-44ac-8685-6570a2efacc4"],
Cell[64002, 1695, 702, 18, 52, "Input",ExpressionUUID->"73cdbb75-7fa5-44dd-869f-631846912996"],
Cell[64707, 1715, 332, 7, 51, "Text",ExpressionUUID->"6558d85e-7311-4b99-9596-3651e826a2fb"],
Cell[65042, 1724, 779, 19, 52, "Input",ExpressionUUID->"1ddbc7d7-ff4e-4522-8645-878c49d98aad"],
Cell[65824, 1745, 384, 8, 51, "Text",ExpressionUUID->"ebb49dda-e5af-4ca2-8e7d-60cc0d5d04a2"],
Cell[66211, 1755, 752, 19, 52, "Input",ExpressionUUID->"43c8e794-bf9f-4713-81cf-ee68fa4e1333"]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[67024, 1781, 237, 4, 69, "Chapter",ExpressionUUID->"c46e9978-8d33-47f4-ae83-b0915ecf9f1b"],
Cell[67264, 1787, 332, 6, 51, "Text",ExpressionUUID->"5dcd7607-dca9-4fca-8c9a-8d7dac0fe916"],
Cell[67599, 1795, 1437, 37, 50, "Input",ExpressionUUID->"494ddb28-4101-45a9-8b30-ac2caa82a41d"],
Cell[69039, 1834, 1160, 30, 29, "Input",ExpressionUUID->"94bafb7c-3e87-45ba-9b78-dba974ee9fe2"],
Cell[70202, 1866, 315, 7, 29, "Input",ExpressionUUID->"66a5b282-e8c5-4e95-b9f2-0ab48ba1ffd6"],
Cell[70520, 1875, 322, 5, 51, "Text",ExpressionUUID->"715da64c-4ad6-4836-9f65-18ed765621f5"],
Cell[CellGroupData[{
Cell[70867, 1884, 477, 9, 29, "Input",ExpressionUUID->"9b3c04fe-b48d-46af-b892-04c9331c5a5e"],
Cell[71347, 1895, 222, 5, 23, "Print",ExpressionUUID->"6b84037e-7aba-4445-b134-3b9d7f7493e6"]
}, Open  ]],
Cell[71584, 1903, 298, 4, 51, "Text",ExpressionUUID->"59a13f53-159d-46ae-8c58-96dbb5603c6a"],
Cell[CellGroupData[{
Cell[71907, 1911, 383, 8, 29, "Input",ExpressionUUID->"2f071340-8e4e-4e0a-83f5-77c6d5ee035c"],
Cell[72293, 1921, 201, 4, 23, "Print",ExpressionUUID->"c6036198-48cd-4854-a725-8f7d2c9f347d"]
}, Open  ]],
Cell[CellGroupData[{
Cell[72531, 1930, 166, 3, 54, "Subsection",ExpressionUUID->"6ab55a5d-18c5-4e93-ace3-829f8426d2ee"],
Cell[72700, 1935, 812, 22, 51, "Code",ExpressionUUID->"8142d618-fa20-4b85-b810-506e1b674602"],
Cell[73515, 1959, 1264, 32, 130, "Input",ExpressionUUID->"c3c53afe-b7f6-4e06-bb21-56df2d291853"],
Cell[74782, 1993, 260, 5, 51, "Text",ExpressionUUID->"dfd4cb66-4b6c-41a9-af6b-8dbbc2992d62"],
Cell[75045, 2000, 585, 14, 29, "Input",ExpressionUUID->"39b24af6-f495-4a35-8e63-95e9819e44f5"],
Cell[75633, 2016, 255, 5, 51, "Text",ExpressionUUID->"ac45fa38-0ab9-44d8-b4af-98ce671b239a"],
Cell[75891, 2023, 1832, 51, 29, "Input",ExpressionUUID->"db4965c6-acae-4dd1-b124-0af8977843c7"],
Cell[77726, 2076, 253, 5, 51, "Text",ExpressionUUID->"fcf9e3fe-37e9-418a-93a6-c111cdfc6201"],
Cell[77982, 2083, 786, 21, 29, "Input",ExpressionUUID->"2e10a4af-f38a-4d70-9c45-84c24ebd6e70"],
Cell[78771, 2106, 318, 7, 51, "Text",ExpressionUUID->"b1c43e5e-9bec-4428-b9fc-1efcbdfc9243"],
Cell[CellGroupData[{
Cell[79114, 2117, 307, 7, 29, "Input",ExpressionUUID->"d836e27a-9075-41c8-9dec-eb0423807b16"],
Cell[79424, 2126, 221, 5, 33, "Output",ExpressionUUID->"ba590d77-6bfa-4c36-8458-f5dc815073c1"]
}, Open  ]],
Cell[CellGroupData[{
Cell[79682, 2136, 308, 7, 29, "Input",ExpressionUUID->"a128d2f9-6f7d-4757-b11f-4a513cfbae7c"],
Cell[79993, 2145, 220, 5, 33, "Output",ExpressionUUID->"dfd62353-130f-4ec2-9fc9-1eba2011b888"]
}, Open  ]],
Cell[80228, 2153, 351, 7, 51, "Text",ExpressionUUID->"284e6285-bf17-4231-91df-1428f879ecc6"],
Cell[CellGroupData[{
Cell[80604, 2164, 356, 8, 29, "Input",ExpressionUUID->"430be553-3653-4ba7-ad53-d41fc1ddbbf4"],
Cell[80963, 2174, 248, 6, 33, "Output",ExpressionUUID->"e13ed3eb-8fae-4b5f-a827-62d9c3ad9aa4"]
}, Open  ]],
Cell[CellGroupData[{
Cell[81248, 2185, 357, 8, 29, "Input",ExpressionUUID->"fdfdae3e-5079-43e9-a1de-7d4fc37366c3"],
Cell[81608, 2195, 245, 6, 33, "Output",ExpressionUUID->"2c4dbd29-25c2-4d6f-b8e9-3d2ffa97d845"]
}, Open  ]],
Cell[81868, 2204, 202, 3, 51, "Text",ExpressionUUID->"5478263b-3def-4c6f-8279-77dd211dd64d"],
Cell[82073, 2209, 780, 23, 50, "Input",ExpressionUUID->"bd670a4d-8793-4280-a3da-4a4627e150cd"],
Cell[CellGroupData[{
Cell[82878, 2236, 162, 3, 45, "Subsubsection",ExpressionUUID->"9a52463c-4221-49b9-b2e7-d166328fe523"],
Cell[83043, 2241, 586, 14, 29, "Input",ExpressionUUID->"cb455290-3bd5-406e-92b3-0f0ce0e8de27"],
Cell[83632, 2257, 309, 7, 31, "Input",ExpressionUUID->"b3f0b43a-2a5e-4dd4-ad1f-945bdf53c703"]
}, Open  ]],
Cell[CellGroupData[{
Cell[83978, 2269, 179, 2, 45, "Subsubsection",ExpressionUUID->"4654884e-5c04-4e9d-86f7-8d5c099dcb3e"],
Cell[84160, 2273, 312, 4, 51, "Text",ExpressionUUID->"0ab49d45-52d1-466e-bed5-fbfce27ee2f9"],
Cell[84475, 2279, 868, 22, 52, "Input",ExpressionUUID->"5807a433-3c64-4219-9132-639b019da95a"],
Cell[85346, 2303, 408, 8, 51, "Text",ExpressionUUID->"eec3dc5e-b370-4e26-be83-214010f74f8a"],
Cell[85757, 2313, 922, 23, 52, "Input",ExpressionUUID->"a9486e34-5135-48ea-8b64-7cc1ebab9b2b"],
Cell[86682, 2338, 412, 8, 51, "Text",ExpressionUUID->"3f97d01e-213f-4e7d-8ed3-69387f3547f8"],
Cell[87097, 2348, 954, 23, 52, "Input",ExpressionUUID->"113760a1-21d4-45e4-bb62-992b3938be81"],
Cell[88054, 2373, 459, 9, 51, "Text",ExpressionUUID->"040b9f80-a710-4f0c-80d0-cff0ab3f17e0"],
Cell[88516, 2384, 1004, 25, 52, "Input",ExpressionUUID->"712a3c84-5f88-4a37-8f68-af4454118e48"]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}
]
*)

