include("basic/gamsworld.jl")

@testset "basic tests" begin
    @testset "no objective and start value" begin
        println("==================================")
        println("No objective and start value")
        println("==================================")
        juniper = DefaultTestSolver(log_levels = [:Table])
        m = Model(optimizer_with_attributes(Juniper.Optimizer, juniper...))
        @variable(m, x, Int, start = 3)
        @constraint(m, x >= 0)
        @constraint(m, x <= 5)
        @NLconstraint(m, x^2 >= 17)
        optimize!(m)
        rand_num = rand()
        optimize!(m)
        @test rand() != rand_num
        @test JuMP.termination_status(m) == MOI.LOCALLY_SOLVED
        @test JuMP.primal_status(m) == MOI.FEASIBLE_POINT
        @test JuMP.dual_status(m) == MOI.NO_SOLUTION
        @test isapprox(JuMP.value(x), 5, atol = sol_atol)
        @test result_count(m) == 1
        @test unsafe_backend(m).inner.primal_start[1] == 3
    end

    @testset "bruteforce" begin
        println("==================================")
        println("Bruteforce")
        println("==================================")
        m = Model()
        @variable(m, 1 <= x[1:4] <= 5, Int)
        special_minimizer_fct(x) = x
        grad(x) = 1.0
        grad2(x) = 0.0
        JuMP.register(
            m,
            :special_minimizer_fct,
            1,
            special_minimizer_fct,
            grad,
            grad2,
        )
        @NLobjective(m, Min, special_minimizer_fct(x[1]))
        juniper_all_solutions = DefaultTestSolver(
            branch_strategy = :StrongPseudoCost,
            list_of_solutions = true,
            strong_restart = true,
            debug = true,
        )
        JuMP.set_optimizer(
            m,
            optimizer_with_attributes(
                Juniper.Optimizer,
                juniper_all_solutions...,
            ),
        )
        @constraint(m, x[1] >= 0.9)
        @constraint(m, x[1] <= 1.1)
        @NLconstraint(m, (x[1] - x[2])^2 >= 0.1)
        @NLconstraint(m, (x[2] - x[3])^2 >= 0.1)
        @NLconstraint(m, (x[1] - x[3])^2 >= 0.1)
        @NLconstraint(m, (x[1] - x[4])^2 >= 0.1)
        @NLconstraint(m, (x[2] - x[4])^2 >= 0.1)
        @NLconstraint(m, (x[3] - x[4])^2 >= 0.1)
        JuMP.set_optimizer_attribute(m, "all_solutions", true)
        JuMP.optimize!(m)
        innermodel = JuMP.unsafe_backend(m).inner
        @test getnstate(innermodel.debugDict, :Integral) == 24
        @test different_hashes(innermodel.debugDict) == true
        counter_test(innermodel.debugDict, Juniper.getnbranches(innermodel))
        list_of_solutions = Juniper.getsolutions(innermodel)
        @test length(unique(list_of_solutions)) == result_count(m)
        @test JuMP.termination_status(m) == MOI.LOCALLY_SOLVED
        @test result_count(m) == 24
    end

    @testset "bruteforce optimizer without attributes" begin
        m = Model(Juniper.Optimizer)
        set_optimizer_attribute(m, "nl_solver", Ipopt.Optimizer)
        set_optimizer_attribute(m, "mip_solver", HiGHS.Optimizer)
        @variable(m, 1 <= x[1:4] <= 5, Int)
        @objective(m, Min, x[1])
        @constraint(m, x[1] >= 0.9)
        @constraint(m, x[1] <= 1.1)
        @NLconstraint(m, (x[1] - x[2])^2 >= 0.1)
        @NLconstraint(m, (x[2] - x[3])^2 >= 0.1)
        @NLconstraint(m, (x[1] - x[3])^2 >= 0.1)
        @NLconstraint(m, (x[1] - x[4])^2 >= 0.1)
        @NLconstraint(m, (x[2] - x[4])^2 >= 0.1)
        @NLconstraint(m, (x[3] - x[4])^2 >= 0.1)
        mktemp() do path, io
            out = stdout
            err = stderr
            redirect_stdout(io)
            redirect_stderr(io)
            status = try
                optimize!(m)
            catch e
                e
            end
            flush(io)
            redirect_stdout(out)
            redirect_stderr(err)
            return
        end
        @test length(sort(unique(convert.(Int, JuMP.value.(x))))) == 4
        @test all(1 .<= JuMP.value.(x) .<= 5)
        @test JuMP.value(x[1]) ≈ 1.0
    end

    @testset "bruteforce full strong w/o restart" begin
        println("==================================")
        println("Bruteforce full strong w/o restart")
        println("==================================")
        juniper_all_solutions = DefaultTestSolver(
            branch_strategy = :StrongPseudoCost,
            all_solutions = true,
            list_of_solutions = true,
            strong_branching_perc = 100,
            strong_branching_nsteps = 100,
            strong_restart = false,
        )
        m = Model(
            optimizer_with_attributes(
                Juniper.Optimizer,
                juniper_all_solutions...,
            ),
        )
        @variable(m, 1 <= x[1:4] <= 5, Int)
        @objective(m, Min, x[1])
        @constraint(m, x[1] >= 0.9)
        @constraint(m, x[1] <= 1.1)
        @NLconstraint(m, (x[1] - x[2])^2 >= 0.1)
        @NLconstraint(m, (x[2] - x[3])^2 >= 0.1)
        @NLconstraint(m, (x[1] - x[3])^2 >= 0.1)
        @NLconstraint(m, (x[1] - x[4])^2 >= 0.1)
        @NLconstraint(m, (x[2] - x[4])^2 >= 0.1)
        @NLconstraint(m, (x[3] - x[4])^2 >= 0.1)
        optimize!(m)
        list_of_solutions = Juniper.getsolutions(unsafe_backend(m).inner)
        @test length(unique(list_of_solutions)) == result_count(m)
        @test termination_status(m) == MOI.LOCALLY_SOLVED
        @test result_count(m) == 24
    end

    @testset "bruteforce approx time limit" begin
        println("==================================")
        println("Bruteforce  approx time limit")
        println("==================================")
        juniper_all_solutions = DefaultTestSolver(
            branch_strategy = :StrongPseudoCost,
            strong_branching_approx_time_limit = 2,
            all_solutions = true,
            list_of_solutions = true,
            strong_restart = true,
        )
        m = Model(
            optimizer_with_attributes(
                Juniper.Optimizer,
                juniper_all_solutions...,
            ),
        )
        @variable(m, 1 <= x[1:4] <= 5, Int)
        @objective(m, Min, x[1])
        @constraint(m, x[1] >= 0.9)
        @constraint(m, x[1] <= 1.1)
        @NLconstraint(m, (x[1] - x[2])^2 >= 0.1)
        @NLconstraint(m, (x[2] - x[3])^2 >= 0.1)
        @NLconstraint(m, (x[1] - x[3])^2 >= 0.1)
        @NLconstraint(m, (x[1] - x[4])^2 >= 0.1)
        @NLconstraint(m, (x[2] - x[4])^2 >= 0.1)
        @NLconstraint(m, (x[3] - x[4])^2 >= 0.1)
        optimize!(m)
        list_of_solutions = Juniper.getsolutions(unsafe_backend(m).inner)
        @test length(unique(list_of_solutions)) == result_count(m)
        @test termination_status(m) == MOI.LOCALLY_SOLVED
        @test result_count(m) == 24
    end

    @testset "bruteforce time limit reliable" begin
        println("==================================")
        println("Bruteforce  time reliable")
        println("==================================")
        juniper_all_solutions = DefaultTestSolver(
            branch_strategy = :Reliability,
            strong_branching_time_limit = 0.02,
            reliability_branching_perc = 100,
            all_solutions = true,
            list_of_solutions = true,
            strong_restart = true,
        )
        optimizer = optimizer_with_attributes(
            Juniper.Optimizer,
            juniper_all_solutions...,
        )
        m = Model(optimizer)
        # default => 1
        @test MOI.supports(backend(m), MOI.NumberOfThreads()) == true
        JuMP.set_optimizer_attribute(m, MOI.NumberOfThreads(), nothing)
        @variable(m, 1 <= x[1:4] <= 5, Int)
        @objective(m, Min, x[1])
        @constraint(m, x[1] >= 0.9)
        @constraint(m, x[1] <= 1.1)
        @NLconstraint(m, (x[1] - x[2])^2 >= 0.1)
        @NLconstraint(m, (x[2] - x[3])^2 >= 0.1)
        @NLconstraint(m, (x[1] - x[3])^2 >= 0.1)
        @NLconstraint(m, (x[1] - x[4])^2 >= 0.1)
        @NLconstraint(m, (x[2] - x[4])^2 >= 0.1)
        @NLconstraint(m, (x[3] - x[4])^2 >= 0.1)
        optimize!(m)
        list_of_solutions = Juniper.getsolutions(unsafe_backend(m).inner)
        @test length(unique(list_of_solutions)) == result_count(m)
        @test JuMP.get_optimizer_attribute(m, MOI.NumberOfThreads()) == 1
        # all solutions are saved => nsolutions should equal length(solutions)
        @test result_count(m) == result_count(m)
        @test termination_status(m) == MOI.LOCALLY_SOLVED
        @test result_count(m) == 24
    end

    @testset "bruteforce PseudoCost" begin
        println("==================================")
        println("Bruteforce PseudoCost")
        println("==================================")
        juniper_all_solutions = DefaultTestSolver(
            branch_strategy = :PseudoCost,
            all_solutions = true,
            list_of_solutions = true,
            strong_restart = true,
        )
        m = Model(
            optimizer_with_attributes(
                Juniper.Optimizer,
                juniper_all_solutions...,
            ),
        )
        @variable(m, 1 <= x[1:4] <= 5, Int)
        @objective(m, Min, x[1])
        @constraint(m, x[1] >= 0.9)
        @constraint(m, x[1] <= 1.1)
        @NLconstraint(m, (x[1] - x[2])^2 >= 0.1)
        @NLconstraint(m, (x[2] - x[3])^2 >= 0.1)
        @NLconstraint(m, (x[1] - x[3])^2 >= 0.1)
        @NLconstraint(m, (x[1] - x[4])^2 >= 0.1)
        @NLconstraint(m, (x[2] - x[4])^2 >= 0.1)
        @NLconstraint(m, (x[3] - x[4])^2 >= 0.1)
        optimize!(m)
        status = termination_status(m)
        println("Status: ", status)
        list_of_solutions = Juniper.getsolutions(unsafe_backend(m).inner)
        @test length(unique(list_of_solutions)) == result_count(m)
        @test status == MOI.LOCALLY_SOLVED
        @test result_count(m) == 24
    end

    @testset "bruteforce Reliability" begin
        println("==================================")
        println("Bruteforce Reliability")
        println("==================================")
        juniper_all_solutions = DefaultTestSolver(
            branch_strategy = :Reliability,
            all_solutions = true,
            list_of_solutions = true,
        )
        m = Model(
            optimizer_with_attributes(
                Juniper.Optimizer,
                juniper_all_solutions...,
            ),
        )
        @variable(m, 1 <= x[1:4] <= 5, Int)
        @objective(m, Min, x[1])
        @constraint(m, x[1] >= 0.9)
        @constraint(m, x[1] <= 1.1)
        @NLconstraint(m, (x[1] - x[2])^2 >= 0.1)
        @NLconstraint(m, (x[2] - x[3])^2 >= 0.1)
        @NLconstraint(m, (x[1] - x[3])^2 >= 0.1)
        @NLconstraint(m, (x[1] - x[4])^2 >= 0.1)
        @NLconstraint(m, (x[2] - x[4])^2 >= 0.1)
        @NLconstraint(m, (x[3] - x[4])^2 >= 0.1)
        optimize!(m)
        status = termination_status(m)
        println("Status: ", status)
        list_of_solutions = Juniper.getsolutions(unsafe_backend(m).inner)
        @test length(unique(list_of_solutions)) == result_count(m)
        @test status == MOI.LOCALLY_SOLVED
        @test result_count(m) == 24
    end

    @testset "no integer" begin
        println("==================================")
        println("no integer")
        println("==================================")
        m = Model(
            optimizer_with_attributes(
                Juniper.Optimizer,
                juniper_strong_restart...,
            ),
        )
        println("Create variables/constr/obj")
        @variable(m, 1 <= x <= 5, start = 2.7)
        @variable(m, -2 <= y <= 2)
        @objective(m, Min, -x - y)
        @NLconstraint(m, y == 2 * cos(2 * x))
        println("before solve")
        optimize!(m)
        status = termination_status(m)
        println("Status: ", status)
        @test status == MOI.LOCALLY_SOLVED
    end

    @testset "infeasible cos" begin
        println("==================================")
        println("Infeasible cos")
        println("==================================")
        m = Model(
            optimizer_with_attributes(
                Juniper.Optimizer,
                juniper_strong_restart...,
            ),
        )
        @variable(m, 1 <= x <= 5, Int)
        @variable(m, -2 <= y <= 2, Int)
        @objective(m, Min, -x - y)
        @NLconstraint(m, y == 2 * cos(2 * x))
        optimize!(m)
        status = termination_status(m)
        println("Status: ", status)
        @test status == MOI.LOCALLY_INFEASIBLE
        @test JuMP.termination_status(m) == MOI.LOCALLY_INFEASIBLE
        @test JuMP.primal_status(m) == MOI.UNKNOWN_RESULT_STATUS
        @test JuMP.dual_status(m) == MOI.NO_SOLUTION
        @test isnan(relative_gap(m))
    end

    @testset "infeasible int reliable" begin
        println("==================================")
        println("Infeasible int reliable")
        println("==================================")
        m = Model(
            optimizer_with_attributes(
                Juniper.Optimizer,
                juniper_reliable_restart...,
            ),
        )
        @variable(m, 1 <= x <= 5, Int)
        @variable(m, -2 <= y <= 2, Int)
        @objective(m, Min, -x - y)
        @NLconstraint(m, y >= sqrt(2))
        @NLconstraint(m, y <= sqrt(3))
        optimize!(m)
        status = termination_status(m)
        println("Status: ", status)
        @test status == MOI.LOCALLY_INFEASIBLE
        @test isnan(relative_gap(m))
    end

    @testset "infeasible sin with different bounds" begin
        println("==================================")
        println("Infeasible  sin with different bounds")
        println("==================================")
        m = Model(
            optimizer_with_attributes(
                Juniper.Optimizer,
                DefaultTestSolver(
                    branch_strategy = :MostInfeasible,
                    feasibility_pump = true,
                    time_limit = 1,
                    mip_solver = optimizer_with_attributes(
                        HiGHS.Optimizer,
                        "output_flag" => false,
                    ),
                )...,
            ),
        )
        @variable(m, x, Int, start = 3)
        @variable(m, y >= 2, Int)
        @constraint(m, 0 <= x <= 5)
        @objective(m, Min, -x - y)
        @NLconstraint(m, y == sin(x))
        optimize!(m)
        @test termination_status(m) == MOI.LOCALLY_INFEASIBLE
    end

    @testset "infeasible relaxation" begin
        println("==================================")
        println("Infeasible relaxation")
        println("==================================")
        m = Model(
            optimizer_with_attributes(
                Juniper.Optimizer,
                DefaultTestSolver(; debug = true)...,
            ),
        )
        @variable(m, 0 <= x[1:10] <= 2, Int)
        @objective(m, Min, sum(x))
        @constraint(m, sum(x[1:5]) <= 20)
        @NLconstraint(m, x[1] * x[2] * x[3] >= 10)
        optimize!(m)
        status = termination_status(m)
        debug1 = unsafe_backend(m).inner.debugDict
        m = Model(
            optimizer_with_attributes(
                Juniper.Optimizer,
                DefaultTestSolver(; debug = true)...,
            ),
        )
        @variable(m, 0 <= x[1:10] <= 2, Int)
        @objective(m, Min, sum(x))
        @constraint(m, sum(x[1:5]) <= 20)
        @NLconstraint(m, x[1] * x[2] * x[3] >= 10)
        optimize!(m)
        status = termination_status(m)
        debug2 = unsafe_backend(m).inner.debugDict
        opts = unsafe_backend(m).inner.options
        # should be deterministic
        @test debug1[:relaxation][:nrestarts] ==
              debug2[:relaxation][:nrestarts] ==
              opts.num_resolve_root_relaxation
        for i in 1:debug1[:relaxation][:nrestarts]
            @test debug1[:relaxation][:restarts][i] ==
                  debug2[:relaxation][:restarts][i]
        end
        println("Status: ", status)
        @test status == MOI.LOCALLY_INFEASIBLE
    end

    @testset "infeasible relaxation 2" begin
        println("==================================")
        println("Infeasible relaxation 2")
        println("==================================")
        m = Model(
            optimizer_with_attributes(
                Juniper.Optimizer,
                juniper_strong_no_restart...,
            ),
        )
        @variable(m, x[1:3], Int)
        @variable(m, y)
        @objective(m, Max, sum(x))
        @NLconstraint(m, x[1]^2 + x[2]^2 + x[3]^2 + y^2 <= 3)
        @NLconstraint(m, x[1]^2 * x[2]^2 * x[3]^2 * y^2 >= 10)
        optimize!(m)
        status = termination_status(m)
        println("Status: ", status)
        @test status == MOI.LOCALLY_INFEASIBLE
    end

    @testset "infeasible integer" begin
        println("==================================")
        println("Infeasible integer")
        println("==================================")
        m = Model(
            optimizer_with_attributes(
                Juniper.Optimizer,
                juniper_strong_no_restart...,
            ),
        )
        @variable(m, 0 <= x[1:10] <= 2, Int)
        @objective(m, Min, sum(x))
        @constraint(m, sum(x[1:5]) <= 20)
        @NLconstraint(m, x[1] * x[2] * x[3] >= 7)
        @NLconstraint(m, x[1] * x[2] * x[3] <= 7.5)
        optimize!(m)
        status = termination_status(m)
        println("Status: ", status)
        @test status == MOI.LOCALLY_INFEASIBLE
    end

    @testset "infeasible in strong" begin
        println("==================================")
        println("Infeasible in strong")
        println("==================================")
        m = Model(
            optimizer_with_attributes(
                Juniper.Optimizer,
                juniper_strong_no_restart...,
            ),
        )
        @variable(m, 0 <= x[1:5] <= 2, Int)
        @objective(m, Min, sum(x))
        @NLconstraint(m, x[3]^2 <= 2)
        @NLconstraint(m, x[3]^2 >= 1.2)
        optimize!(m)
        status = termination_status(m)
        println("Status: ", status)
        @test status == MOI.LOCALLY_INFEASIBLE
    end

    @testset "One Integer small Reliable" begin
        println("==================================")
        println("One Integer small Reliable")
        println("==================================")
        m = Model()
        @variable(m, x >= 0, Int, start = 2)
        @variable(m, y >= 0)
        @variable(m, 0 <= u <= 10, Int)
        @variable(m, w == 1)
        myf(x, y) = -3x - y
        function ∇f(g, x, y)
            g[1] = -3
            g[2] = -1
            return
        end
        JuMP.register(m, :myf, 2, myf, ∇f)
        @NLobjective(m, Min, myf(x, y))
        JuMP.set_optimizer(
            m,
            optimizer_with_attributes(
                Juniper.Optimizer,
                juniper_reliable_restart...,
            ),
        )
        @constraint(m, 3x + 10 <= 20)
        @NLconstraint(m, y^2 <= u * w)
        optimize!(m)
        println("Obj: ", JuMP.objective_value(m))
        println("x: ", JuMP.value(x))
        println("y: ", JuMP.value(y))
        @test termination_status(m) == MOI.LOCALLY_SOLVED
        @test isapprox(JuMP.objective_value(m), -12.162277, atol = opt_atol)
        @test isapprox(JuMP.value(x), 3, atol = sol_atol)
        @test isapprox(JuMP.value(y), 3.162277, atol = sol_atol)
    end

    @testset "One Integer small Strong" begin
        println("==================================")
        println("One Integer small Strong")
        println("==================================")
        m = Model(
            optimizer_with_attributes(
                Juniper.Optimizer,
                juniper_strong_no_restart...,
            ),
        )
        @variable(m, x >= 0, Int)
        @variable(m, y >= 0)
        @variable(m, 0 <= u <= 10, Int)
        @variable(m, w == 1)
        @objective(m, Min, -3x - y)
        @constraint(m, 3x + 10 <= 20)
        @NLconstraint(m, y^2 <= u * w)
        optimize!(m)
        println("Obj: ", JuMP.objective_value(m))
        println("x: ", JuMP.value(x))
        println("y: ", JuMP.value(y))
        @test termination_status(m) == MOI.LOCALLY_SOLVED
        @test isapprox(JuMP.objective_value(m), -12.162277, atol = opt_atol)
        @test isapprox(JuMP.value(x), 3, atol = sol_atol)
        @test isapprox(JuMP.value(y), 3.162277, atol = sol_atol)
    end

    @testset "One Integer small MostInfeasible" begin
        println("==================================")
        println("One Integer small MostInfeasible")
        println("==================================")
        m = Model()
        @variable(m, x >= 0, Int)
        @variable(m, y >= 0)
        @variable(m, 0 <= u <= 10, Int)
        @variable(m, w == 1)
        special_minimizer_fct(x, y) = -3x - y
        JuMP.register(
            m,
            :special_minimizer_fct,
            2,
            special_minimizer_fct;
            autodiff = true,
        )
        @NLobjective(m, Min, special_minimizer_fct(x, y))
        @constraint(m, 3x + 10 <= 20)
        @NLconstraint(m, y^2 <= u * w)
        JuMP.set_optimizer(
            m,
            optimizer_with_attributes(
                Juniper.Optimizer,
                DefaultTestSolver(branch_strategy = :MostInfeasible)...,
            ),
        )
        optimize!(m)
        println("Obj: ", JuMP.objective_value(m))
        println("x: ", JuMP.value(x))
        println("y: ", JuMP.value(y))
        @test termination_status(m) == MOI.LOCALLY_SOLVED
        @test isapprox(JuMP.objective_value(m), -12.162277, atol = opt_atol)
        @test isapprox(JuMP.value(x), 3, atol = sol_atol)
        @test isapprox(JuMP.value(y), 3.162277, atol = sol_atol)
    end

    @testset "One Integer small PseudoCost" begin
        println("==================================")
        println("One Integer small PseudoCost")
        println("==================================")
        m = Model(
            optimizer_with_attributes(Juniper.Optimizer, juniper_pseudo...),
        )
        @variable(m, x >= 0, Int)
        @variable(m, y >= 0)
        @variable(m, 0 <= u <= 10, Int)
        @variable(m, w == 1)
        @objective(m, Min, -3x - y)
        @constraint(m, 3x + 10 <= 20)
        @NLconstraint(m, y^2 <= u * w)
        optimize!(m)
        println("Obj: ", JuMP.objective_value(m))
        println("x: ", JuMP.value(x))
        println("y: ", JuMP.value(y))
        @test termination_status(m) == MOI.LOCALLY_SOLVED
        @test isapprox(JuMP.objective_value(m), -12.162277, atol = opt_atol)
        @test isapprox(JuMP.value(x), 3, atol = sol_atol)
        @test isapprox(JuMP.value(y), 3.162277, atol = sol_atol)
    end

    @testset "Three Integers Small Strong" begin
        println("==================================")
        println("Three Integers Small Strong")
        println("==================================")
        m = Model(
            optimizer_with_attributes(
                Juniper.Optimizer,
                juniper_strong_no_restart...,
            ),
        )
        @variable(m, x >= 0, Int)
        @variable(m, y >= 0, Int)
        @variable(m, 0 <= u <= 10, Int)
        @variable(m, w == 1)
        @objective(m, Min, -3x - y)
        @constraint(m, 3x + 10 <= 20)
        @NLconstraint(m, y^2 <= u * w)
        optimize!(m)
        println("Obj: ", JuMP.objective_value(m))
        @test termination_status(m) == MOI.LOCALLY_SOLVED
        @test isapprox(JuMP.objective_value(m), -12, atol = opt_atol)
        @test isapprox(JuMP.value(x), 3, atol = sol_atol)
        @test isapprox(JuMP.value(y), 3, atol = sol_atol)
    end

    @testset "Three Integers Small MostInfeasible" begin
        println("==================================")
        println("Three Integers Small MostInfeasible")
        println("==================================")
        m = Model(
            optimizer_with_attributes(Juniper.Optimizer, juniper_mosti...),
        )
        @variable(m, x >= 0, Int)
        @variable(m, y >= 0, Int)
        @variable(m, 0 <= u <= 10, Int)
        @variable(m, w == 1)
        @objective(m, Min, -3x - y)
        @constraint(m, 3x + 10 <= 20)
        @NLconstraint(m, y^2 <= u * w)
        optimize!(m)
        println("Obj: ", JuMP.objective_value(m))
        @test termination_status(m) == MOI.LOCALLY_SOLVED
        @test isapprox(JuMP.objective_value(m), -12, atol = opt_atol)
        @test isapprox(JuMP.value(x), 3, atol = sol_atol)
        @test isapprox(JuMP.value(y), 3, atol = sol_atol)
    end

    @testset "Three Integers Small PseudoCost" begin
        println("==================================")
        println("Three Integers Small PseudoCost")
        println("==================================")
        m = Model(
            optimizer_with_attributes(Juniper.Optimizer, juniper_pseudo...),
        )
        @variable(m, x >= 0, Int)
        @variable(m, y >= 0, Int)
        @variable(m, 0 <= u <= 10, Int)
        @variable(m, w == 1)
        @objective(m, Min, -3x - y)
        @constraint(m, 3x + 10 <= 20)
        @NLconstraint(m, y^2 <= u * w)
        optimize!(m)
        println("Obj: ", JuMP.objective_value(m))
        @test termination_status(m) == MOI.LOCALLY_SOLVED
        @test isapprox(JuMP.objective_value(m), -12, atol = opt_atol)
        @test isapprox(JuMP.value(x), 3, atol = sol_atol)
        @test isapprox(JuMP.value(y), 3, atol = sol_atol)
    end

    @testset "Knapsack Max" begin
        println("==================================")
        println("KNAPSACK")
        println("==================================")
        m = Model(
            optimizer_with_attributes(
                Juniper.Optimizer,
                DefaultTestSolver(;
                    traverse_strategy = :DBFS,
                    incumbent_constr = true,
                    mip_solver = optimizer_with_attributes(
                        HiGHS.Optimizer,
                        "output_flag" => false,
                    ),
                    strong_branching_time_limit = 1,
                )...,
            ),
        )
        v = [10, 20, 12, 23, 42]
        w = [12, 45, 12, 22, 21]
        @variable(m, x[1:5], Bin)
        @objective(m, Max, v' * x)
        @NLconstraint(m, sum(w[i] * x[i]^2 for i in 1:5) <= 45)
        optimize!(m)
        println("Obj: ", JuMP.objective_value(m))
        @test termination_status(m) == MOI.LOCALLY_SOLVED
        @test isapprox(JuMP.objective_value(m), 65, atol = opt_atol)
        @test isapprox(JuMP.objective_bound(m), 65, atol = opt_atol)
        @test isapprox(JuMP.value.(x), [0, 0, 0, 1, 1], atol = sol_atol)
    end

    @testset "Knapsack Max Reliable" begin
        println("==================================")
        println("KNAPSACK Reliable no restart")
        println("==================================")
        m = Model(
            optimizer_with_attributes(
                Juniper.Optimizer,
                DefaultTestSolver(;
                    branch_strategy = :Reliability,
                    strong_restart = false,
                    strong_branching_time_limit = 1,
                    gain_mu = 0.5,
                )...,
            ),
        )
        v = [10, 20, 12, 23, 42]
        w = [12, 45, 12, 22, 21]
        @variable(m, x[1:5], Bin)
        @objective(m, Max, v' * x)
        @NLconstraint(m, sum(w[i] * x[i]^2 for i in 1:5) <= 45)
        optimize!(m)
        println("Obj: ", JuMP.objective_value(m))
        @test termination_status(m) == MOI.LOCALLY_SOLVED
        @test isapprox(JuMP.objective_value(m), 65, atol = opt_atol)
        @test isapprox(JuMP.objective_bound(m), 65, atol = opt_atol)
        @test isapprox(JuMP.value.(x), [0, 0, 0, 1, 1], atol = sol_atol)
    end

    @testset "Integer at root" begin
        println("==================================")
        println("INTEGER AT ROOT")
        println("==================================")
        m = Model(
            optimizer_with_attributes(
                Juniper.Optimizer,
                DefaultTestSolver()...,
            ),
        )
        @variable(m, x[1:6] <= 1, Int)
        @constraint(m, x[1:6] .== 1)
        @objective(m, Max, sum(x))
        @NLconstraint(m, x[1] * x[2] * x[3] + x[4] * x[5] * x[6] <= 100)
        optimize!(m)
        @test termination_status(m) == MOI.LOCALLY_SOLVED
        @test isapprox(JuMP.objective_value(m), 6, atol = opt_atol)
        # objective bound doesn't exists anymore in Ipopt
        #@test isapprox(JuMP.objective_bound(m), 0, atol=opt_atol) # Ipopt return 0
    end

    @testset "Knapsack Max with epsilon" begin
        println("==================================")
        println("KNAPSACK with epsilon")
        println("==================================")
        m = Model(
            optimizer_with_attributes(
                Juniper.Optimizer,
                DefaultTestSolver(;
                    traverse_strategy = :DBFS,
                    obj_epsilon = 0.5,
                )...,
            ),
        )
        v = [10, 20, 12, 23, 42]
        w = [12, 45, 12, 22, 21]
        @variable(m, x[1:5], Bin)
        @objective(m, Max, v' * x)
        @NLconstraint(m, sum(w[i] * x[i]^2 for i in 1:5) <= 45)
        optimize!(m)
        println("Obj: ", JuMP.objective_value(m))
        @test termination_status(m) == MOI.LOCALLY_SOLVED
        @test isapprox(JuMP.objective_value(m), 65, atol = opt_atol)
        @test isapprox(JuMP.value.(x), [0, 0, 0, 1, 1], atol = sol_atol)
    end

    @testset "Knapsack Max with epsilon too strong" begin
        println("==================================")
        println("KNAPSACK with epsilon too strong")
        println("==================================")
        m = Model(
            optimizer_with_attributes(
                Juniper.Optimizer,
                DefaultTestSolver(;
                    traverse_strategy = :DBFS,
                    obj_epsilon = 0.1,
                )...,
            ),
        )
        v = [10, 20, 12, 23, 42]
        w = [12, 45, 12, 22, 21]
        # set all to 1 which is infeasible otherwise incumbent solution would be found
        @variable(m, x[1:5], Bin, start = 1)
        @objective(m, Max, v' * x)
        @NLconstraint(m, sum(w[i] * x[i]^2 for i in 1:5) <= 45)
        optimize!(m)
        @test termination_status(m) == MOI.LOCALLY_INFEASIBLE
    end

    @testset "Batch.mod Restart" begin
        println("==================================")
        println("BATCH.MOD RESTART")
        println("==================================")
        m = batch_problem()
        JuMP.set_optimizer(
            m,
            optimizer_with_attributes(
                Juniper.Optimizer,
                juniper_strong_restart...,
            ),
        )
        optimize!(m)
        @test termination_status(m) == MOI.LOCALLY_SOLVED
        juniper_val = JuMP.objective_value(m)
        println("Solution by Juniper")
        println("obj: ", juniper_val)
        @test isapprox(
            juniper_val,
            285506.5082,
            atol = opt_atol,
            rtol = opt_rtol,
        )
    end

    @testset "Batch.mod Restart 2 Levels" begin
        println("==================================")
        println("BATCH.MOD RESTART 2 LEVELS")
        println("==================================")
        m = batch_problem()
        JuMP.set_optimizer(
            m,
            optimizer_with_attributes(
                Juniper.Optimizer,
                juniper_strong_restart_2...,
            ),
        )
        optimize!(m)
        @test termination_status(m) == MOI.LOCALLY_SOLVED
        juniper_val = JuMP.objective_value(m)
        println("Solution by Juniper")
        println("obj: ", juniper_val)
        @test isapprox(
            juniper_val,
            285506.5082,
            atol = opt_atol,
            rtol = opt_rtol,
        )
    end

    @testset "cvxnonsep_nsig20r.mod restart" begin
        println("==================================")
        println("cvxnonsep_nsig20r.MOD RESTART")
        println("==================================")
        m = cvxnonsep_nsig20r_problem()
        JuMP.set_optimizer(
            m,
            optimizer_with_attributes(
                Juniper.Optimizer,
                juniper_strong_restart...,
            ),
        )
        optimize!(m)
        @test termination_status(m) == MOI.LOCALLY_SOLVED
        juniper_val = JuMP.objective_value(m)
        println("Solution by Juniper")
        println("obj: ", juniper_val)
        @test isapprox(juniper_val, 80.9493, atol = opt_atol, rtol = opt_rtol)
    end

    @testset "cvxnonsep_nsig20r.mod no restart" begin
        println("==================================")
        println("cvxnonsep_nsig20r.MOD NO RESTART")
        println("==================================")
        m = cvxnonsep_nsig20r_problem()
        JuMP.set_optimizer(
            m,
            optimizer_with_attributes(
                Juniper.Optimizer,
                juniper_strong_no_restart...,
            ),
        )
        optimize!(m)
        @test termination_status(m) == MOI.LOCALLY_SOLVED
        juniper_val = JuMP.objective_value(m)
        println("Solution by Juniper")
        println("obj: ", juniper_val)
        @test isapprox(juniper_val, 80.9493, atol = opt_atol, rtol = opt_rtol)
    end

    @testset "Knapsack solution limit and table print test" begin
        println("==================================")
        println("Knapsack solution limit and table print test")
        println("==================================")
        juniper_one_solution = DefaultTestSolver(
            log_levels = [:Table],
            branch_strategy = :MostInfeasible,
            solution_limit = 1,
        )
        m = Model()
        v = [10, 20, 12, 23, 42]
        w = [12, 45, 12, 22, 21]
        @variable(m, x[1:5], Bin)
        @objective(m, Max, v' * x)
        @NLconstraint(m, sum(w[i] * x[i]^2 for i in 1:5) <= 45)
        JuMP.set_optimizer(
            m,
            optimizer_with_attributes(
                Juniper.Optimizer,
                juniper_one_solution...,
            ),
        )
        optimize!(m)
        @test termination_status(m) == MOI.SOLUTION_LIMIT
        # maybe 2 found at the same time
        @test result_count(m) <= 2
        @test result_count(m) >= 1
    end

    @testset "bruteforce obj_epsilon" begin
        println("==================================")
        println("Bruteforce obj_epsilon")
        println("==================================")
        juniper_obj_eps = DefaultTestSolver(
            branch_strategy = :StrongPseudoCost,
            obj_epsilon = 0.4,
        )
        m = Model(
            optimizer_with_attributes(Juniper.Optimizer, juniper_obj_eps...),
        )
        @variable(m, 1 <= x[1:4] <= 5, Int)
        @objective(m, Min, x[1])
        @constraint(m, x[1] >= 0.9)
        @constraint(m, x[1] <= 1.1)
        @NLconstraint(m, (x[1] - x[2])^2 >= 0.1)
        @NLconstraint(m, (x[2] - x[3])^2 >= 0.1)
        @NLconstraint(m, (x[1] - x[3])^2 >= 0.1)
        @NLconstraint(m, (x[1] - x[4])^2 >= 0.1)
        @NLconstraint(m, (x[2] - x[4])^2 >= 0.1)
        @NLconstraint(m, (x[3] - x[4])^2 >= 0.1)
        optimize!(m)
        # maybe 2 found at the same time
        @test termination_status(m) == MOI.LOCALLY_SOLVED
    end

    @testset "bruteforce best_obj_stop not reachable" begin
        println("==================================")
        println("Bruteforce best_obj_stop not reachable")
        println("==================================")
        juniper_best_obj_stop = DefaultTestSolver(
            branch_strategy = :StrongPseudoCost,
            best_obj_stop = 0.8,
        )
        m = Model(
            optimizer_with_attributes(
                Juniper.Optimizer,
                juniper_best_obj_stop...,
            ),
        )
        @variable(m, 1 <= x[1:4] <= 5, Int)
        @objective(m, Min, x[1])
        @constraint(m, x[1] >= 0.9)
        @constraint(m, x[1] <= 1.1)
        @NLconstraint(m, (x[1] - x[2])^2 >= 0.1)
        @NLconstraint(m, (x[2] - x[3])^2 >= 0.1)
        @NLconstraint(m, (x[1] - x[3])^2 >= 0.1)
        @NLconstraint(m, (x[1] - x[4])^2 >= 0.1)
        @NLconstraint(m, (x[2] - x[4])^2 >= 0.1)
        @NLconstraint(m, (x[3] - x[4])^2 >= 0.1)
        optimize!(m)
        # not possible to reach but should be solved anyway
        @test termination_status(m) == MOI.LOCALLY_SOLVED
    end

    @testset "bruteforce best_obj_stop reachable" begin
        println("==================================")
        println("Bruteforce best_obj_stop reachable")
        println("==================================")
        juniper_one_solution = DefaultTestSolver(
            branch_strategy = :StrongPseudoCost,
            best_obj_stop = 1,
        )
        m = Model(
            optimizer_with_attributes(
                Juniper.Optimizer,
                juniper_one_solution...,
            ),
        )
        @variable(m, 1 <= x[1:4] <= 5, Int)
        @objective(m, Min, x[1])
        @constraint(m, x[1] >= 0.9)
        @constraint(m, x[1] <= 1.1)
        @NLconstraint(m, (x[1] - x[2])^2 >= 0.1)
        @NLconstraint(m, (x[2] - x[3])^2 >= 0.1)
        @NLconstraint(m, (x[1] - x[3])^2 >= 0.1)
        @NLconstraint(m, (x[1] - x[4])^2 >= 0.1)
        @NLconstraint(m, (x[2] - x[4])^2 >= 0.1)
        @NLconstraint(m, (x[3] - x[4])^2 >= 0.1)
        optimize!(m)
        # reachable and should break
        @test termination_status(m) == MOI.OBJECTIVE_LIMIT
        # maybe 2 found at the same time
        @test result_count(m) <= 2
        @test result_count(m) >= 1
    end

    # this test has a lot "Only almost locally solved" warnings (in mumps at least)
    @testset "Sum 1/x = 2" begin
        println("==================================")
        println("Sum 1/x = 2")
        println("==================================")
        m = Model(
            optimizer_with_attributes(Juniper.Optimizer, juniper_pseudo...),
        )
        @variable(m, 1 <= x[1:11], Int)
        @constraint(m, [i = 2:11], x[i-1] <= x[i] - 1)
        @NLconstraint(m, sum(1 / x[i] for i in 1:11) == 2)
        optimize!(m)
        status = termination_status(m)
        # in mumps/on travis this returns ALMOST_LOCALLY_SOLVED but with
        # ma27/locally it is LOCALLY_SOLVED
        @test Juniper.state_is_optimal(status; allow_almost = true)
        @test isapprox(2, sum(1 / v for v in JuMP.value.(x)), atol = opt_atol)
    end

    # this test has a lot "Only almost locally solved" warnings
    @testset "Sum 1/x = 2 don't allow almost" begin
        println("==================================")
        println("Sum 1/x = 2 don't allow almost")
        println("==================================")
        m = Model(
            optimizer_with_attributes(
                Juniper.Optimizer,
                DefaultTestSolver(
                    branch_strategy = :PseudoCost,
                    allow_almost_solved = false,
                )...,
            ),
        )
        @variable(m, 1 <= x[1:11], Int)
        @constraint(m, [i = 2:11], x[i-1] <= x[i] - 1)
        @NLconstraint(m, sum(1 / x[i] for i in 1:11) == 2)
        optimize!(m)
        # even without almost this should still be solveable
        @test termination_status(m) == MOI.LOCALLY_SOLVED
        @test isapprox(2, sum(1 / v for v in JuMP.value.(x)), atol = opt_atol)
    end

    @testset "Integer fractional objective w/ HiGHS" begin
        println("==================================")
        println("Integer fractional objective w/ HiGHS")
        println("==================================")
        m = Model(
            optimizer_with_attributes(
                Juniper.Optimizer,
                DefaultTestSolver(
                    mip_solver = optimizer_with_attributes(
                        HiGHS.Optimizer,
                        "output_flag" => false,
                    ),
                )...,
            ),
        )
        @variable(m, 1 <= x <= 5, Int)
        @variable(m, 1 <= y <= 5, Int)
        @objective(m, Max, x / y)
        optimize!(m)
        @test termination_status(m) == MOI.LOCALLY_SOLVED
    end

    #this test has an expression where a variable will be dereferenced twice
    @testset "Nested variable reference" begin
        println("==================================")
        println("Nested variable reference")
        println("==================================")
        m = Model(
            optimizer_with_attributes(
                Juniper.Optimizer,
                DefaultTestSolver()...,
            ),
        )
        x = @variable(m, x[i = 1:5, j = 1:5], Bin)
        xrowsum = @NLexpression(m, xrowsum[i = 1:5], sum(x[i, j] for j in 1:5))
        @NLobjective(
            m,
            Max,
            sum(sum(x[i, j] + xrowsum[i] for i in 1:5) for j in 1:5)
        )
        optimize!(m)
        @test JuMP.termination_status(m) == MOI.LOCALLY_SOLVED
    end

    @testset "Expr dereferencing for @NLexpression" begin
        # See issue 184
        m = Model()
        set_optimizer(
            m,
            optimizer_with_attributes(
                Juniper.Optimizer,
                DefaultTestSolver(
                    mip_solver = optimizer_with_attributes(
                        HiGHS.Optimizer,
                        "output_flag" => false,
                    ),
                )...,
            ),
        )
        @variable(m, 0 <= a_var <= 1)
        @variable(m, bin_var, Int)
        b_expr = @NLexpression(m, bin_var / 1)
        @NLconstraint(m, 1.1 >= b_expr)
        an_expr = @NLexpression(m, a_var / 1)
        @NLobjective(m, Max, bin_var + an_expr)
        optimize!(m)
        @test JuMP.objective_value(m) ≈ 2.0
        @test JuMP.value(bin_var) ≈ 1
        @test JuMP.value(a_var) ≈ 1
    end
end
