# Markdown Backend Examples

```@meta
CurrentModule = PrettyTables
```

```@setup markdown_examples
using PrettyTables
```

Here, we can see some examples of text tables generated by PrettyTables.jl. The `A` object,
when referenced, is defined as:

```julia-repl
julia> A = Any[
    1    false      1.0     0x01
    2     true      2.0     0x02
    3    false      3.0     0x03
    4     true      4.0     0x04
    5    false      5.0     0x05
    6     true      6.0     0x06
]
```

```@setup markdown_examples
A = Any[
    1    false      1.0     0x01
    2     true      2.0     0x02
    3    false      3.0     0x03
    4     true      4.0     0x04
    5    false      5.0     0x05
    6     true      6.0     0x06
]
```

---

```@example markdown_examples
pretty_table(A; backend = :markdown)
```

---

```@example markdown_examples
pretty_table(
  A;
  backend = :markdown,
  style   = MarkdownTableStyle(first_line_column_label = MarkdownStyle(italic = true))
)
```

---

```@example markdown_examples
data = [
    10.0 6.5
     3.0 3.0
     0.1 1.0
]

row_labels = [
    "Atmospheric drag"
    "Gravity gradient"
    "Solar radiation pressure"
]

column_labels = [
    [MultiColumn(2, "Value", :c)],
    [
        "Torque [10⁻⁶ Nm]",
        "Angular Momentum [10⁻³ Nms]"
    ]
]

pretty_table(
    data;
    backend = :markdown,
    column_labels,
    row_labels,
    stubhead_label = "Effect",
    summary_row_labels = ["Total"],
    summary_rows = [(data, i) -> sum(data[:, i])],
)
```

---

```@example markdown_examples
t = 0:1:20

data = hcat(t, ones(length(t) ), t, 0.5.*t.^2);

column_labels = [
    ["Time", "Acceleration", "Velocity", "Distance"],
    [ "[s]",     "[m / s²]",  "[m / s]",      "[m]"]
]

hl_p = MarkdownHighlighter(
    (data, i, j) -> (j == 4) && (data[i, j] > 9),
    MarkdownStyle(italic = true)
)

hl_v = MarkdownHighlighter(
    (data, i, j) -> (j == 3) && (data[i, j] > 9),
    MarkdownStyle(bold = true)
)

pretty_table(
    data;
    backend       = :markdown,
    column_labels = column_labels,
    highlighters  = [hl_p, hl_v],
)
```
